AJS.test.require("com.atlassian.jira.jira-projects-plugin:browseproject", function() {

    module("TabModule", {
        setup: function() {
            this.projectRoot = "http://my.jira:2990/j/browse/FOO";
            this.pathname = "/j/browse/FOO";
            this.history = {
                start: sinon.spy(),
                navigate: sinon.spy()
            };
            sinon.stub(JIRA.Messages, "showErrorMsg");
            sinon.stub(AJS,"contextPath").returns('/j');
            sinon.stub(JIRA.Projects, "TabView", function () {});
            var model = _.extend({}, Backbone.Events);
            this.model = model;
            sinon.stub(JIRA.Projects, "TabModel", function() {
                return model;
            });
            var router = _.extend({}, Backbone.Events);
            sinon.stub(JIRA.Projects, "TabPanelRouter", function() {
                return router;
            });
        },
        teardown: function() {
            JIRA.Projects.TabModel.restore();
            JIRA.Projects.TabView.restore();
            JIRA.Projects.TabPanelRouter.restore();
            AJS.contextPath.restore();
            JIRA.Messages.showErrorMsg.restore();
        }
    });

    test("navigates to tab selection on switch", function () {
        var sourceTabUrl = this.projectRoot + '?selectedTab=the.current.tab';
        var targetTabUrl = this.projectRoot + '?selectedTab=the.target.tab';

        JIRA.Projects.TabLoader.init({}, this.history, {href: sourceTabUrl}, true);
        this.model.trigger("switch", targetTabUrl);

        ok(this.history.navigate.calledOnce, "Navigation occurs on new tab selected");
        ok(this.history.navigate.calledWithExactly("?selectedTab=the.target.tab", false),
                "navigates to tab selection on switch without triggering route");
    });

    test("navigates to legacy tab selection on switch, preserving query string params", function () {
        var sourceTabUrl = this.projectRoot + 'http://my.jira:2990/secure/BrowseProject.jspa?id=123&selectedTab=the.current.tab';
        var targetTabUrl = this.projectRoot + 'http://my.jira:2990/secure/BrowseProject.jspa?id=123&selectedTab=the.target.tab';

        JIRA.Projects.TabLoader.init({}, this.history, {href: sourceTabUrl}, true);
        this.model.trigger("switch", targetTabUrl);

        equal(this.history.navigate.callCount, 1, "Navigation occurs on new tab selected");
        ok(this.history.navigate.calledWithExactly("?id=123&selectedTab=the.target.tab", false),
                "navigates to legacy tab selection without triggering route");
    });

    test("sets up router with new model", function () {
        var location = {href: this.projectRoot + "?selectedTab=tab.key", pathname: this.pathname};
        JIRA.Projects.TabLoader.init({}, this.history, location, true);
        ok(JIRA.Projects.TabPanelRouter.getCall(0).args[0] === this.model);
        ok(JIRA.Projects.TabPanelRouter.getCall(0).args[1] === location);
    });

    test("starts the history with the pathname", function () {
        JIRA.Projects.TabLoader.init({}, this.history, {href: this.projectRoot + "?selectedTab=tab.key", pathname: this.pathname}, true);
        equal(this.history.start.getCall(0).args[0].root, "/j/browse/FOO");
    });

    test("saves the currently selected tab into the url if it's missing (for nav purposes)", function () {
        var links = '<ul class="aui-page-panel-nav"><li class="active"><a href="http://j.url/browse/KEY?selectedTab=this.tab" class="browse-tab"></a></li>' +
                '<li><a href="http://jira.url/browse/PROJECTKEY?selectedTab=not.this.tab" class="browse-tab"></a></li>';
        var $el = jQuery('<div>' + links + '<div id="project-tab"></div></div>');
        JIRA.Projects.TabLoader.init($el, this.history,
                {href: this.projectRoot, pathname: this.pathname}, true);
        equal(this.history.navigate.callCount, 1, "Updates url");
        equal(this.history.navigate.getCall(0).args[0], "?selectedTab=this.tab", "Updates url");
        equal(this.history.navigate.getCall(0).args[1].trigger, false, "Updates url without triggering a route");
        equal(this.history.navigate.getCall(0).args[1].replace, true, "Updates url without adding history");
    });

    test("saves the currently selected tab into the legacy url if it's missing (for nav purposes)", function () {
        var links = '<ul class="aui-page-panel-nav"><li class="active"><a href="http://j.url/browse/KEY?selectedTab=this.tab" class="browse-tab"></a></li>' +
                '<li><a href="http://jira.url/browse/PROJECTKEY?selectedTab=not.this.tab" class="browse-tab"></a></li>';
        var $el = jQuery('<div>' + links + '<div id="project-tab"></div></div>');
        JIRA.Projects.TabLoader.init($el, this.history,
                {href: 'http://my.jira:2990/secure/BrowseProject.jspa?id=123',
                    pathname: '/secure/BrowseProject.jspa'}, true);
        equal(this.history.navigate.callCount, 1, "Updates url");
        equal(this.history.navigate.getCall(0).args[0], "?id=123&selectedTab=this.tab", "Updates url");
        equal(this.history.navigate.getCall(0).args[1].trigger, false, "Updates url without triggering a route");
        equal(this.history.navigate.getCall(0).args[1].replace, true, "Updates url without adding history");
    });

    test("triggers a route if there is a hash fragment present", function () {
        JIRA.Projects.TabLoader.init({}, this.history,
                {href: 'http://my.jira:2990/secure/BrowseProject.jspa?selectedTab=this.tab&id=123#?selectedTab=that.tab',
                    pathname: '/secure/BrowseProject.jspa'}, true);
        equal(this.history.start.callCount, 1, "Starts the history");
        equal(this.history.start.getCall(0).args[0].silent, false, "triggers a route");
    });

    test("saves the current url if the browser does not have pushState or a hash fragment", function () {
        var links = '<ul class="aui-page-panel-nav"><li class="active"><a href="http://j.url/browse/KEY?selectedTab=this.tab" class="browse-tab"></a></li>' +
                '<li><a href="http://jira.url/browse/PROJECTKEY?selectedTab=not.this.tab" class="browse-tab"></a></li>';
        var $el = jQuery('<div>' + links + '<div id="project-tab"></div></div>');
        JIRA.Projects.TabLoader.init($el, this.history,
                {href: 'http://my.jira:2990/secure/BrowseProject.jspa?id=1000&selectedTab=this.tab', pathName: '/secure/BrowseProject.jspa'});
        equal(this.history.navigate.callCount, 1, "Updates the url");
    });

    test("does not save the current url if the browser lacks pushState but has a hash fragment", function () {
        var links = '<ul class="aui-page-panel-nav"><li class="active"><a href="http://j.url/browse/KEY?selectedTab=this.tab" class="browse-tab"></a></li>' +
                '<li><a href="http://jira.url/browse/PROJECTKEY?selectedTab=not.this.tab" class="browse-tab"></a></li>';
        var $el = jQuery('<div>' + links + '<div id="project-tab"></div></div>');
        JIRA.Projects.TabLoader.init($el, this.history,
                {href: 'http://my.jira:2990/secure/BrowseProject.jspa?id=1000&selectedTab=this.tab#selectedTab=that.tab',
                 pathName: '/secure/BrowseProject.jspa'});
        equal(this.history.navigate.callCount, 0, "Does not update the url");
    });

    test("displays error message on tab not found", function () {
        var location = {href: this.projectRoot + "?selectedTab=tab.key"};

        JIRA.Projects.TabLoader.init({}, this.history, location, true);
        this.model.trigger("switch-error", {status: 404});

        equal(JIRA.Messages.showErrorMsg.callCount, 1, "Error message displayed on 404 when switching tabs");
        equal(JIRA.Messages.showErrorMsg.getCall(0).args[0], JIRA.Projects.ErrorMessages.notFound(), "Not found message displayed when tab not found");
    });

    test("displays error message on tab load timeout", function () {
        var location = {href: this.projectRoot + "?selectedTab=tab.key"};

        JIRA.Projects.TabLoader.init({}, this.history, location, true);
        this.model.trigger("switch-error", {statusText: "timeout"});

        equal(JIRA.Messages.showErrorMsg.callCount, 1, "Error message displayed on timeout error when switching tabs");
        equal(JIRA.Messages.showErrorMsg.getCall(0).args[0], JIRA.Projects.ErrorMessages.timeout(), "Not found message displayed when loading of tab times out");
    });

    test("displays error message on tab load error", function () {
        var location = {href: this.projectRoot + "?selectedTab=tab.key"};

        JIRA.Projects.TabLoader.init({}, this.history, location, true);
        this.model.trigger("switch-error", {});

        equal(JIRA.Messages.showErrorMsg.callCount, 1, "Error message displayed on error when switching tabs");
        equal(JIRA.Messages.showErrorMsg.getCall(0).args[0], JIRA.Projects.ErrorMessages.genericError(), "Not found message displayed when loading of tab encounters an error");
    });

    test("redirects to login on authentication failure", function () {
        var location = { href: this.projectRoot + "?selectedTab=tab.key", replace: sinon.spy() };

        JIRA.Projects.TabLoader.init({}, this.history, location, true);
        this.model.trigger("switch-error", { loggedOut: true }, "backhere");

        equal(location.replace.callCount, 1, "Redirected when session has expired");

        var loginUrl = AJS.contextPath() + "/login.jsp?permissionViolation=true&os_destination=backhere";
        equal(location.replace.getCall(0).args[0], loginUrl, "Redirected to logon page when session has expired");
    });

});

