package com.atlassian.jira.security.groups;

import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.OperationNotPermittedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.exception.embedded.InvalidGroupException;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.Page;
import com.atlassian.jira.util.PageRequest;

import java.util.Collection;
import javax.annotation.Nonnull;

import javax.annotation.Nullable;

/**
 * This object can be used to manage groups in JIRA.
*
 * @since v3.13
 */
public interface GroupManager
{
    /**
     * Returns <code>true</code> if the given group name exists.
     *
     * @param groupName The group name.
     * @return <code>true</code> if the given group name exists.
     *
     * @since v3.13
     */
    boolean groupExists(String groupName);

    /**
     * Returns <code>true</code> if the given group exists.
     *
     * @param group The group.
     * @return <code>true</code> if the given group exists.
     *
     * @since v7.0
     */
    boolean groupExists(@Nonnull Group group);

    /**
     * Get all groups.
     *
     * @return Collection of all Groups.
     * @since v4.3
     * @deprecated Since v7.0. Only retrieve the users you really need. See {@link com.atlassian.jira.bc.group.search.GroupPickerSearchService}
     */
    @Deprecated
    Collection<Group> getAllGroups();

    /**
     * Create a group with the given name.
     *
     * @param groupName The group name.
     * @return the newly created Group.
     * @throws InvalidGroupException if the group already exists in ANY associated directory or the group template does not have the required properties populated.
     * @throws com.atlassian.crowd.exception.OperationNotPermittedException if the directory has been configured to not allow the operation to be performed
     *
     * @since v4.3
     */
    Group createGroup(String groupName) throws OperationNotPermittedException, InvalidGroupException;

    /**
     * Returns the Group for this groupName, else null if no such Group exists.
     *
     * @param groupName The group name.
     * @return The Group for this groupName, else null if no such Group exists.
     */
    Group getGroup(String groupName);

    /**
     * Returns the Group for this groupName, if no such Group exists then a proxy unknown immutable Group object is
     * returned.
     *
     * @param groupName The group name.
     * @return The Group for this groupName.
     */
    Group getGroupEvenWhenUnknown(String groupName);

    /**
     * Returns the Group for this groupName, else null if no such Group exists.
     * <p/>
     * Legacy synonym for {@link #getGroup(String)}.
     *
     * @param groupName The group name.
     * @return The Group for this groupName, else null if no such Group exists.
     * @see #getGroup(String)
     * @deprecated use {@link #getGroup(String)} instead. Since 6.5
     */
    @Deprecated
    Group getGroupObject(String groupName);

    /**
     * Returns {@code true} if the user is a member of the group.
     * <p>
     * Note that if the username or groupname is null, then it will return false.
     * <p>
     * <strong>Performance note</strong>: If you have a full user object, then calling one of the other
     * {@code isUserInGroup()} methods directly is more efficient.
     *
     * @param username user to inspect.
     * @param groupname group to inspect.
     * @return {@code true} if and only if the user is a direct or indirect (nested) member of the group.
     * @deprecated Use one of the other {@code isUserInGroup} methods that takes a concrete user object instead.
     *      Since v6.4.8.
     * @see #isUserInGroup(ApplicationUser, Group)
     * @see #isUserInGroup(ApplicationUser, String)
     */
    boolean isUserInGroup(@Nullable final String username, @Nullable final String groupname);

    /**
     * Returns {@code true} if the user is a member of the group.
     * <p>
     * Note that if the User or Group object is null, then it will return false.
     * This was done to retain consistency with the old OSUser behaviour of User.inGroup() and Group.containsUser()
     *
     * @param user user to inspect.
     * @param group group to inspect.
     * @return {@code true} if and only if the user is a direct or indirect (nested) member of the group.
     * @since v6.4.8
     */
    boolean isUserInGroup(@Nullable ApplicationUser user, @Nullable Group group);

    /**
     * Returns {@code true} if the user is a member of the named group.
     * <p>
     * If you already have the user object, then this method is faster than the alternative of passing in the username
     * because it saves on an unnecessary user lookup to find the correct User Directory.
     *
     * @param user user to inspect.
     * @param groupName group to inspect.
     * @return {@code true} if and only if the user is a direct or indirect (nested) member of the group.
     * @since v6.4.8
     */
    boolean isUserInGroup(@Nullable ApplicationUser user, @Nullable String groupName);

    /**
     * Returns all the users in a group.
     * 
     * @param groupName The group
     * @return all the users that belongs to the group.
     * @since v4.3
     *
     * @see {@link #getUsersInGroup(Group)}
     */
    Collection<ApplicationUser> getUsersInGroup(String groupName);

    /**
     * Returns a page with users in a group. Users are sorted by name in ascending order.
     *
     * @param groupName name of the group for which users are returned
     * @param includeInactive if set to true inactive users will be returned as well
     * @param pageRequest parameters of the page to return
     * @return page with the users
     */
    Page<ApplicationUser> getUsersInGroup(final String groupName, final Boolean includeInactive, final PageRequest pageRequest);

    /**
     * Returns all the users in a group.
     * This will include nested group members.
     *
     * @param group The group
     * @return all the users that belongs to the group.
     * @since v4.3
     *
     * @see {@link #getUsersInGroup(String)}
     *
     * @throws NullPointerException if the group is null
     */
    Collection<ApplicationUser> getUsersInGroup(Group group);

    /**
     * Returns a count of all active users in a group.
     * This will include nested group members.
     *
     * @param group The group
     * @return a count of all the users that belongs to the group.
     * @since 7.0
     *
     * @see {@link #getUsersInGroupCount(String)}
     */
    int getUsersInGroupCount(Group group);

    /**
     * Returns a count of all active users in a group.
     * This will include nested group members.
     *
     * @param groupName The group name
     * @return a count of all the users that belongs to the group.
     * @since 7.0
     *
     * @see {@link #getUsersInGroupCount(Group)}
     */
    int getUsersInGroupCount(String groupName);

    /**
     * Returns the names of all the users in a group.
     * This will include nested group members.
     *
     * @param group The group
     * @return all the users that belongs to the group.
     * @since v5.0
     *
     * @see {@link #getUsersInGroup(Group)}
     *
     * @throws NullPointerException if the group is null
     */
    Collection<String> getUserNamesInGroup(Group group);

    /**
     * Returns the names of all the users in a group.
     * This will include nested group members.
     *
     * @param groupName The group
     * @return all the users that belongs to the group.
     * @since v5.0
     *
     * @see {@link #getUsersInGroup(String)}
     *
     * @throws NullPointerException if the group is null
     */
    Collection<String> getUserNamesInGroup(String groupName);

    /**
     * Returns all the users that are direct members of the group.
     * This will NOT include nested group members.
     *
     * @param group The group
     * @return all the users that belongs to the group.
     * @since v4.3
     *
     * @see {@link #getUsersInGroup(String)}
     *
     * @throws NullPointerException if the group is null
     */
    Collection<ApplicationUser> getDirectUsersInGroup(Group group);

    /**
     * Returns all the groups that the given user belongs to.
     * @param userName The user
     * @return all the groups that the given user belongs to.
     * @since v4.3
     * @see #getGroupNamesForUser(String)
     */
    Collection<Group> getGroupsForUser(String userName);

    /**
     * Returns all the groups that the given user belongs to.
     * @param userName The user
     * @return all the groups that the given user belongs to.
     * @since v4.3
     * @see #getGroupNamesForUser(String)
     */
    Collection<Group> getGroupsForUser(ApplicationUser userName);

    /**
     * Returns the names of all the groups that the given user belongs to.
     * @param userName The user
     * @return all the groups that the given user belongs to.
     * @since v4.3
     * @see #getGroupsForUser(String)
     * @see #getGroupNamesForUser(com.atlassian.jira.user.ApplicationUser)
     */
    Collection<String> getGroupNamesForUser(String userName);

    /**
     * Returns the names of all the groups that the given user belongs to.
     * @param user The user
     * @return all the groups that the given user belongs to.
     * @since v6.2.5
     * @see #getGroupsForUser(String)
     */
    Collection<String> getGroupNamesForUser(@Nonnull ApplicationUser user);

    /**
     * Adds a user as a member of a group.
     *
     * @param user  The user that will become a member of the group.
     * @param group The group that will gain a new member.
     * @throws UserNotFoundException  if the {@code user} could not be found
     * @throws GroupNotFoundException if the {@code group} could not be found
     * @throws OperationNotPermittedException if the directory has been configured to not allow the operation to be performed
     * @throws OperationFailedException If the underlying directory implementation failed to execute the operation.
     *
     * @since v4.3
     */
    void addUserToGroup(ApplicationUser user, Group group) throws GroupNotFoundException, UserNotFoundException, OperationNotPermittedException, OperationFailedException;
}
