/*
 * Copyright (c) 2002-2004
 * All rights reserved.
 */

package com.atlassian.jira.mail;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.issue.Issue;

import javax.annotation.Nullable;
import javax.mail.Message;

@PublicApi
public interface MailThreadManager
{
    /**
     * Indicates an action in response to an incoming email.
     * @since v5.2.3
     */
    public enum MailAction {ISSUE_CREATED_FROM_EMAIL, ISSUE_COMMENTED_FROM_EMAIL}

    /**
     * Remembers the given incoming MessageID.
     * <p>
     * This is used when someone emails JIRA and CCs some else, and that person in turn does a reply-all. Such an email
     * will not have the Issue Key in the subject.
     *
     * @param messageId The incoming Message-ID
     * @param senderAddress The sender
     * @param issue the issue that was affected (created or commented)
     * @param action Issue created or Issue commented
     *
     * @since v5.2.3
     */
    public void storeIncomingMessageId(String messageId, String senderAddress, Issue issue, MailAction action);

    /**
     * Thread the given email which is related to the given issue.
     *
     * @param email the email to be sent
     * @param issue the issue that the email is about
     */
    public void threadNotificationEmail(Email email, Issue issue);

    /**
     * Looks for an issue associated with the given message by inspecting the "In-Reply-To" header of the message.
     * <p>
     * Notifications sent from JIRA have a special form that allows us to parse out the Issue ID.
     * We also remember the incoming Message-IDs so we can tell if another recipient replies to that message.
     *
     * @param message message to analyse
     * @return associated issue or null if no issue is associated with this message.
     */
    @Nullable
    public Issue getAssociatedIssueObject(Message message);

    /**
     * Looks for an issue associated with the given message by inspecting the "Message-ID" header of the message.
     *
     * @param messageId Message-ID to be checked
     * @return Issue that is already associated with this Message-ID or null if none
     */
    @Nullable
    public Issue findIssueFromMessageId(String messageId);

    /**
     * Removes rows from NotificationInstance table associated with the given issue.
     * Used when we delete an issue.
     * @param issueId the issue
     * @return row count
     */
    public int removeAssociatedEntries(Long issueId);
}
