package com.atlassian.jira.plugin.issuetabpanel;

import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import com.google.common.collect.ImmutableList;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;
import java.util.Arrays;
import java.util.List;

/**
 * Reply object used in the {@link IssueTabPanel2} SPI. This class is immutable as long as the IssueActions that it
 * contains are also immutable.
 * <p/>
 * Example usage:
 * <pre>
 *     return GetActionsReply.create(myPluginActions);
 * </pre>
 *
 * @see IssueTabPanel2
 * @since v5.0
 */
@PublicApi
@Immutable
final public class GetActionsReply
{
    /**
     * Creates a new GetActionsReply containing a single action.
     *
     * @param action the IssueAction that the response will contain
     * @return a new GetActionsReply
     */
    public static GetActionsReply create(@Nonnull IssueAction... action)
    {
        return create(Arrays.asList(action));
    }

    /**
     * Creates a new GetActionsReply containing multiple actions.
     *
     * @param actions the IssueActions that the response will contain
     * @return a new GetActionsReply
     */
    public static GetActionsReply create(@Nonnull Iterable<? extends IssueAction> actions)
    {
        return new GetActionsReply(actions);
    }

    /**
     * The actions.
     */
    private final ImmutableList<IssueAction> actions;

    /**
     * Creates an new GetActionsReply.
     *
     * @param actions the IssueAction instances to add to the created GetActionsReply
     */
    private GetActionsReply(@Nonnull Iterable<? extends IssueAction> actions)
    {
        this.actions = ImmutableList.copyOf(actions);
    }

    /**
     * @return a list of IssueAction, or null
     */
    @Nonnull
    @Internal
    public List<IssueAction> actions()
    {
        return actions;
    }
}
