package com.atlassian.jira.jql.context;

import com.atlassian.crowd.embedded.api.User;
import com.atlassian.jira.jql.operand.JqlOperandResolver;
import com.atlassian.jira.jql.operand.QueryLiteral;
import com.atlassian.jira.jql.operator.OperatorClasses;
import com.atlassian.jira.jql.resolver.VersionIndexInfoResolver;
import com.atlassian.jira.jql.resolver.VersionResolver;
import com.atlassian.jira.jql.util.JqlVersionPredicate;
import com.atlassian.jira.project.version.Version;
import com.atlassian.jira.security.PermissionManager;
import com.atlassian.jira.util.Predicate;
import com.atlassian.jira.util.collect.CollectionUtil;
import com.atlassian.query.clause.TerminalClause;
import com.atlassian.query.operator.Operator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static com.atlassian.jira.util.dbc.Assertions.notNull;

/**
 * A context factory for project version clauses (fix and affects). The contexts that are generated by this factory will all be
 * {@link com.atlassian.jira.jql.context.QueryContextElementType#IMPLICIT}. The context will contain any projects
 * that the version is relevant for and an {@link com.atlassian.jira.jql.context.AllIssueTypesContext}.
 *
 * @since v4.0
 */
public class VersionClauseContextFactory extends AbstractProjectAttributeClauseContextFactory<Version> implements ClauseContextFactory
{
    private final JqlOperandResolver jqlOperandResolver;
    private final VersionResolver versionResolver;

    public VersionClauseContextFactory(final JqlOperandResolver jqlOperandResolver, final VersionResolver versionResolver,
            final PermissionManager permissionManager)
    {
        super(new VersionIndexInfoResolver(versionResolver), jqlOperandResolver, permissionManager);
        this.jqlOperandResolver = notNull("jqlOperandResolver", jqlOperandResolver);
        this.versionResolver = notNull("versionResolver", versionResolver);
    }

    ClauseContext getContextFromClause(final User searcher, final TerminalClause terminalClause)
    {
        final Operator operator = terminalClause.getOperator();
        if (!handlesOperator(operator))
        {
            return ClauseContextImpl.createGlobalClauseContext();
        }

        final List<QueryLiteral> literals = jqlOperandResolver.getValues(searcher, terminalClause.getOperand(), terminalClause);
        // Run through and figure out all the versions that were specified by the user
        final List<Version> specifiedVersions = new ArrayList<Version>();

        if (literals != null)
        {
            for (QueryLiteral literal : literals)
            {
                if (!literal.isEmpty())
                {
                    final List<Long> ids = getIds(literal);
                    for (Long id : ids)
                    {
                        specifiedVersions.add(versionResolver.get(id));
                    }
                }
            }
        }

        if (!specifiedVersions.isEmpty())
        {
            final Collection<Version> contextVersions;

            // If the operator is a negation operator than we need to get the set of versions that are NOT the
            // specified versions.
            if (isRelationalOperator(operator))
            {
                contextVersions = new ArrayList<Version>();
                // Run through all the specified versions and get the ones depending on the relational operator
                for (Version specifiedVersion : specifiedVersions)
                {
                    Predicate<Version> predicate = new JqlVersionPredicate(operator, specifiedVersion);
                    for (Version matchedVersion : CollectionUtil.filter(versionResolver.getAll(), predicate))
                    {
                        contextVersions.add(matchedVersion);
                    }
                }
            }
            else if (isNegationOperator(operator))
            {
                contextVersions = new ArrayList<Version>(versionResolver.getAll());
                contextVersions.removeAll(specifiedVersions);
            }
            else
            {
                contextVersions = specifiedVersions;
            }

            final Set<ProjectIssueTypeContext> contexts = new HashSet<ProjectIssueTypeContext>();
            // Lets add the contexts for all the versions we have found
            for (Version contextVersion : contextVersions)
            {
                contexts.addAll(getContextsForProject(searcher, contextVersion.getProjectObject()));
            }

            return new ClauseContextImpl(contexts);
        }
        else
        {
            // if we somehow got no versions or only empties, just return the global context
            return ClauseContextImpl.createGlobalClauseContext();
        }
    }

    private boolean handlesOperator(Operator operator)
    {
        return OperatorClasses.EQUALITY_OPERATORS_WITH_EMPTY.contains(operator)
                || OperatorClasses.RELATIONAL_ONLY_OPERATORS.contains(operator);
    }
}
