package com.atlassian.jira.event.issue;

import com.atlassian.crowd.embedded.api.User;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.comments.Comment;
import com.atlassian.jira.issue.worklog.Worklog;
import org.ofbiz.core.entity.GenericValue;

import java.util.Map;

/**
 * Component responsible for dispatching issue events.
 *
 * @since v4.4
 */
public interface IssueEventManager
{

    /**
     * Dispatch event of given type, configuring whether or not a mail notification should be sent (useful e.g. for bulk
     * edit).
     *
     * @param eventTypeId type of event
     * @param issue affected issue
     * @param remoteUser user initiating the event
     * @param sendMail whether or not a mail notification should be sent
     */
    void dispatchEvent(Long eventTypeId, Issue issue, User remoteUser, boolean sendMail);

    /**
     * Dispatch event of given type with custom parameters.
     *
     * @param eventTypeId type of event
     * @param issue affected issue
     * @param params custom event parameters
     * @param remoteUser user initiating the event
     * @param sendMail whether or not a mail notification should be sent
     */
    void dispatchEvent(Long eventTypeId, Issue issue, Map<String,Object> params, User remoteUser, boolean sendMail);

    /**
     * Dispatch event of given type with custom parameters.
     *
     * @param eventTypeId type of event
     * @param issue  affected issue
     * @param params custom event parameters
     * @param remoteUser user initiating the event
     */
    void dispatchEvent(Long eventTypeId, Issue issue, Map params, User remoteUser);

    /**
     * Dispatch event of given type.
     *
     * @param eventTypeId  type of event
     * @param issue        affected issue
     * @param remoteUser   remoteUser user initiating the event
     * @param comment      comment for this event
     * @param worklog      worklog for this event
     * @param changelog    An attached changeGroup for this event
     */
    void dispatchEvent(Long eventTypeId, Issue issue, User remoteUser, Comment comment, Worklog worklog, GenericValue changelog);

    /**
     * Dispatch event of given type.
     *
     * @param eventTypeId  type of event
     * @param issue        affected issue
     * @param remoteUser   remoteUser user initiating the event
     * @param comment      comment for this event
     * @param worklog      worklog for this event
     * @param changelog    attached changeGroup for this event
     * @param sendMail     whether or not a mail notification should be sent
     */
    void dispatchEvent(Long eventTypeId, Issue issue, User remoteUser, Comment comment, Worklog worklog, GenericValue changelog, boolean sendMail);

    /**
     * Dispatch event of given type with custom parameters.
     *
     * @param eventTypeId  type of event
     * @param issue        affected issue
     * @param remoteUser   remoteUser user initiating the event
     * @param comment      comment for this event
     * @param worklog      worklog for this event
     * @param changelog    attached changeGroup for this event
     * @param params       custom event parameters
     */
    void dispatchEvent(Long eventTypeId, Issue issue, User remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params);

    /**
     * Dispatch event of given type with custom parameters.
     *
     * @param eventTypeId  type of event
     * @param issue        affected issue
     * @param remoteUser   remoteUser user initiating the event
     * @param comment      comment for this event
     * @param worklog      worklog for this event
     * @param changelog    attached changeGroup for this event
     * @param params       custom event parameters
     * @param sendMail     whether or not a mail notification should be sent
     */
    void dispatchEvent(Long eventTypeId, Issue issue, User remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params, boolean sendMail);

    /**
     * Dispatch event of given type.
     *
     * @param eventTypeId  type of event
     * @param issue        affected issue
     * @param remoteUser   remoteUser user initiating the event
     * @param changelog    attached changeGroup for this event
     * @param sendMail     whether or not a mail notification should be sent
     * @param subtasksUpdated  if subtask have been modified.
     */
    void dispatchEvent(Long eventTypeId, Issue issue, User remoteUser, GenericValue changelog, boolean sendMail, boolean subtasksUpdated);

    /**
     * Dispatch event of given type with custom parameters.
     *
     * @param eventTypeId   type of event
     * @param issue         affected issue
     * @param remoteUser    remoteUser user initiating the event
     * @param comment       comment for this event
     * @param worklog       worklog for this event
     * @param changelog     attached changeGroup for this event
     * @param params        custom event parameters
     * @param sendMail      whether or not a mail notification should be sent
     * @param subtasksUpdated if subtask have been modified.
     */
    void dispatchEvent(Long eventTypeId, Issue issue, User remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params, boolean sendMail, boolean subtasksUpdated);
}
