package com.atlassian.jconnect.rest.resources;

import com.atlassian.jconnect.jira.JMCProjectService;
import com.atlassian.jconnect.jira.UserHelper;
import com.atlassian.jconnect.rest.entities.AdminForm;
import com.atlassian.jconnect.rest.entities.ProjectStatusEntity;
import com.atlassian.jconnect.util.Either;
import com.atlassian.jira.bc.ServiceOutcome;
import com.atlassian.jira.bc.project.ProjectService;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.security.PermissionManager;
import com.atlassian.jira.security.Permissions;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.json.JSONException;
import com.atlassian.jira.util.json.JSONObject;
import com.atlassian.jira.util.lang.Pair;
import com.atlassian.sal.api.user.UserManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.List;

/**
 * This resource should only be accesible by jira admins.
 * It provides methods for enabling/disabling JMC for a specific JIRA Project.
 * There are also methods to query the active number of JMCified projects.
 */
@Path("/admin")
public class JMCAdminResource {

    private ProjectService jiraProjectService;
    private JMCProjectService jmcProjectService;
    private UserManager userManager;
    private UserHelper userHelper;
    private PermissionManager permissionManager;
    private static final Logger log = LoggerFactory.getLogger(JMCAdminResource.class);
    
    public JMCAdminResource(PermissionManager permissionManager,
                            UserManager userManager,
                            UserHelper userHelper,
                            ProjectService jiraProjectService,
                            JMCProjectService projectService) {
        this.jiraProjectService = jiraProjectService;
        this.jmcProjectService = projectService;
        this.userManager = userManager;
        this.userHelper = userHelper;
        this.permissionManager = permissionManager;
    }


    /**
     *  Returns a ProjectStatus object, indicating whether or not JMC is enabled.disabled,
     *  keys are active/inactive etc
     * **/
    @GET
    @Path("projectStatus")
    @Produces(MediaType.APPLICATION_JSON)
    public Response projectStatus(@QueryParam("projectId") Long projectId) throws Exception {

        final Either<Pair<Project, ApplicationUser>, Response.ResponseBuilder> result = validateProject(projectId);
        if (result.getRight() != null) {
            return result.getRight().build();
        }
        final Project project = result.getLeft().first();

        final boolean enabled = jmcProjectService.isJiraConnectProject(project);
        final String apiKey = enabled ? jmcProjectService.lookupApiKeyFor(project) : "";

        final boolean reindexRequired = false; // re-index only required on state change
        final boolean apiKeyActive = jmcProjectService.isApiKeyEnabledFor(project);
        final boolean crashesEnabled = jmcProjectService.isCrashesEnabledFor(project);

        return Response.ok(new ProjectStatusEntity(project.getId(), project.getKey(), enabled, apiKey, reindexRequired, apiKeyActive, crashesEnabled)).build();
    }

    @POST
    @Path("toggleOnOff")
    @Produces(MediaType.APPLICATION_JSON)
    public Response enableProject(AdminForm form) throws Exception {


        final Either<Pair<Project, ApplicationUser>, Response.ResponseBuilder> result = validateProject(form.getProjectId());
        if (result.getRight() != null) {
            return result.getRight().build();
        }
        
        final ApplicationUser user = result.getLeft().second();
        final Project project = result.getLeft().first();

        final boolean enabled = jmcProjectService.toggleForJiraConnect(project);
        final String apiKey = enabled ? jmcProjectService.generateOrRetrieveAPIKeyFor(project) : "";

        final boolean reindexRequired = jmcProjectService.reindexRequiredFor(project, user);
        final boolean apiKeyActive = jmcProjectService.isApiKeyEnabledFor(project);
        final boolean crashesEnabled = jmcProjectService.isCrashesEnabledFor(project);

        //TODO: prompt user to add Location to all screens ? all view Screens ?
        return Response.ok(new ProjectStatusEntity(project.getId(), project.getKey(), enabled, apiKey, reindexRequired, apiKeyActive,crashesEnabled)).build();
    }

    @POST
    @Path("resetApiKey")
    @Produces(MediaType.APPLICATION_JSON)
    public Response resetApiKey(AdminForm form) throws Exception {
        final Either<Pair<Project, ApplicationUser>, Response.ResponseBuilder> result = validateProject(form.getProjectId());
        if (result.getRight() != null) {
            return result.getRight().build();
        }
        final Project project = result.getLeft().first();
        final String newApiKey = jmcProjectService.generateApiKeyFor(project);
        final boolean enabled = jmcProjectService.isJiraConnectProject(project);
        final boolean crashesEnabled = jmcProjectService.isCrashesEnabledFor(project);
        final boolean apiKeyActive = jmcProjectService.isApiKeyEnabledFor(project);
        return Response.ok(new ProjectStatusEntity(form.getProjectId(), project.getKey(), enabled, newApiKey, false, apiKeyActive, crashesEnabled)).build();
    }

    @POST
    @Path("setApiKeyActiveStatus")
    @Produces(MediaType.APPLICATION_JSON)
    public Response toggleApiKey(AdminForm form) throws Exception {
        final Either<Pair<Project, ApplicationUser>, Response.ResponseBuilder> result = validateProject(form.getProjectId());
        if (result.getRight() != null) {
            return result.getRight().build();
        }
        final Project project = result.getLeft().first();
        final boolean jiraConnectProject = jmcProjectService.isJiraConnectProject(project);
        final boolean crashesEnabled = jmcProjectService.isCrashesEnabledFor(project);
        final boolean apiKeyActive = jmcProjectService.toggleApiKeyFor(project, form.isActive());
        return Response.ok(new ProjectStatusEntity(form.getProjectId(), project.getKey(), jiraConnectProject, null, false, apiKeyActive, crashesEnabled)).build();
    }

    @POST
    @Path("setCrashesEnabledStatus")
    @Produces(MediaType.APPLICATION_JSON)
    public Response toggleCrashesEnabledStatus(AdminForm form) throws Exception {
        final Either<Pair<Project, ApplicationUser>, Response.ResponseBuilder> result = validateProject(form.getProjectId());
        if (result.getRight() != null) {
            return result.getRight().build();
        }
        final Project project = result.getLeft().first();
        final boolean jiraConnectProject = jmcProjectService.isJiraConnectProject(project);
        final boolean apiKeyActive = jmcProjectService.isApiKeyEnabledFor(project);

        final boolean crashesEnabled = jmcProjectService.toggleCrashesFor(project, form.isCrashesEnabled());
        return Response.ok(new ProjectStatusEntity(form.getProjectId(), project.getKey(), jiraConnectProject, null, false, apiKeyActive, crashesEnabled)).build();
    }


    @GET
    @Path("stats")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getStatistics() {
        final String remoteUsername = userManager.getRemoteUsername();
        final ApplicationUser user = userHelper.getApplicationUser(remoteUsername);
        if (!userManager.isSystemAdmin(remoteUsername)) {
            return Response.status(Response.Status.UNAUTHORIZED).entity("Only System Administrators may access this resource").build();
        }
        final ServiceOutcome<List<Project>> allProjects = this.jiraProjectService.getAllProjects(user);
        final List<Project> projects = allProjects.getReturnedValue();
        int jmcProjectCount = 0;
        for (Project project : projects) {
            jmcProjectCount += jmcProjectService.isJiraConnectProject(project) ? 1 : 0;
        }
        final long jmcIssueCount = jmcProjectService.countJMCIssues(user);
        final JSONObject json = new JSONObject();
        try {
            json.put("projectCount", jmcProjectCount);
            json.put("issueCount", jmcIssueCount);
            return Response.ok(json.toString(), MediaType.APPLICATION_JSON_TYPE).build();
        } catch (JSONException e) {
            log.error(e.getMessage(), e);
            return Response.serverError().entity(e.getMessage()).build();
        }
    }
    
    private Either<Pair<Project, ApplicationUser>, Response.ResponseBuilder> validateProject(Long projectId) {

        final String remoteUsername = userManager.getRemoteUsername();
        if (projectId == null) {
            return Either.right(Response.status(Response.Status.BAD_REQUEST).entity("Missing projectId parameter in request"));
        }
        final ApplicationUser remoteUser = userHelper.getApplicationUser(remoteUsername);
        final Project project = jiraProjectService.getProjectById(remoteUser, projectId).getProject();

        if (project == null) {
            return Either.right(Response.status(Response.Status.BAD_REQUEST).entity("Project with ID: " + projectId + " not found"));
        }

        if (remoteUser == null || !userHasAdminPermission(remoteUsername, project, remoteUser))
        {
            return Either.right(Response.status(Response.Status.UNAUTHORIZED).entity("Permission denied: Admins only."));
        }
        return Either.left(Pair.of(project, remoteUser));
    }

    private boolean userHasAdminPermission(String remoteUsername, Project project, ApplicationUser user) {
        return  permissionManager.hasPermission(Permissions.PROJECT_ADMIN, project, user) ||
                permissionManager.hasPermission(Permissions.ADMINISTER, user) ||
                userManager.isSystemAdmin(remoteUsername);
    }


}
