package com.atlassian.jconnect.jira.tabpanel;

import com.atlassian.jconnect.jira.customfields.BuiltInField;
import com.atlassian.jconnect.jira.customfields.CustomFieldHelper;
import com.atlassian.jconnect.jira.customfields.IssueLocation;
import com.atlassian.jconnect.jira.customfields.Location;
import com.atlassian.jira.bc.issue.search.SearchService;
import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.search.SearchException;
import com.atlassian.jira.issue.search.SearchResults;
import com.atlassian.jira.project.browse.BrowseContext;
import com.atlassian.jira.security.JiraAuthenticationContext;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.ErrorCollection;
import com.atlassian.jira.util.SimpleErrorCollection;
import com.atlassian.jira.web.bean.PagerFilter;
import com.atlassian.velocity.VelocityManager;
import com.google.common.base.Function;
import com.google.common.collect.Lists;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Project Tab panel fragment with Google Maps summary.
 *
 */
public class GoogleMapsFragment extends AbstractFragment
{

    private final CustomFieldHelper customFieldHelper;
    private final SearchService searchService;
    private static final int MAX_LATEST_ISSUES = 500; // only report on the MAX_LATEST_ISSUES number of issues

    public GoogleMapsFragment(VelocityManager velocityManager, ApplicationProperties applicationProperites,
                              JiraAuthenticationContext jiraAuthenticationContext, CustomFieldHelper customFieldHelper,
                              SearchService searchService)
    {
        super(velocityManager, applicationProperites, jiraAuthenticationContext);
        this.customFieldHelper = customFieldHelper;
        this.searchService = searchService;
    }

    public Map<String, Object> createVelocityParams(BrowseContext ctx)
    {
        final Map<String, Object> params = new HashMap<String, Object>();
        final ErrorCollection errors = new SimpleErrorCollection();
        final List<IssueLocation> issues = collectIssueLocations(ctx, errors);
        params.put("errors", errors);
        params.put("issues", issues);
        params.put("firstIssue", issues.isEmpty() ? null : issues.get(0));
        return params;
    }

    private List<IssueLocation> collectIssueLocations(BrowseContext ctx, ErrorCollection errors)
    {
        final String jql = "project=" + ctx.getProject().getKey() + " order by createdDate";
        final ApplicationUser loggedInUser = getJiraAuthenticationContext().getLoggedInUser();
        SearchService.ParseResult parseResult = searchService.parseQuery(loggedInUser, jql);
        if (!parseResult.isValid())
        {
            errors.addErrorMessages(parseResult.getErrors().getErrorMessages());
        }
        else
        {
            try
            {
                final SearchResults results = searchService.search(loggedInUser,
                        parseResult.getQuery(), PagerFilter.newPageAlignedFilter(0, MAX_LATEST_ISSUES));
                return Lists.transform(results.getIssues(), new Function<Issue, IssueLocation>()
                {
                    public IssueLocation apply(@Nullable Issue from) {
                        if (from != null) {
                            Location location = (Location) from.getCustomFieldValue(getLocationField());
                            return location != null ? new IssueLocation(location, from) : null;
                        }
                        return null;
                    }
                });
            }
            catch (SearchException se)
            {
                errors.addErrorMessage(se.getLocalizedMessage());
            }
        }
        return Collections.emptyList();
    }

    public String getId()
    {
        return "googlemaps";
    }

    public boolean showFragment(BrowseContext browseContext)
    {
        return customFieldHelper.containsField(browseContext.getProject(), getLocationField());
    }

    private CustomField getLocationField()
    {
        return customFieldHelper.initBuiltInCustomField(BuiltInField.LOCATION);
    }
}
