package com.atlassian.jconnect.jira.customfields;

import com.atlassian.jira.JiraDataTypes;
import com.atlassian.jira.issue.customfields.CustomFieldSearcher;
import com.atlassian.jira.issue.customfields.SingleValueCustomFieldValueProvider;
import com.atlassian.jira.issue.customfields.converters.DoubleConverter;
import com.atlassian.jira.issue.customfields.searchers.AbstractInitializationCustomFieldSearcher;
import com.atlassian.jira.issue.customfields.searchers.CustomFieldSearcherClauseHandler;
import com.atlassian.jira.issue.customfields.searchers.SimpleCustomFieldSearcherClauseHandler;
import com.atlassian.jira.issue.customfields.searchers.information.CustomFieldSearcherInformation;
import com.atlassian.jira.issue.customfields.searchers.renderer.CustomFieldRenderer;
import com.atlassian.jira.issue.customfields.searchers.transformer.CustomFieldInputHelper;
import com.atlassian.jira.issue.customfields.searchers.transformer.ExactTextCustomFieldSearchInputTransformer;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.index.indexers.FieldIndexer;
import com.atlassian.jira.issue.search.ClauseNames;
import com.atlassian.jira.issue.search.searchers.information.SearcherInformation;
import com.atlassian.jira.issue.search.searchers.renderer.SearchRenderer;
import com.atlassian.jira.issue.search.searchers.transformer.SearchInputTransformer;
import com.atlassian.jira.jql.operand.JqlOperandResolver;
import com.atlassian.jira.security.JiraAuthenticationContext;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.jira.web.FieldVisibilityManager;
import com.atlassian.query.operator.Operator;

import java.util.Collections;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import static com.google.common.base.Preconditions.checkState;

/**
 * {@link com.atlassian.jira.issue.customfields.CustomFieldSearcher} for location custom field.
 *
 * @since v4.4
 */
public class LocationSearcher extends AbstractInitializationCustomFieldSearcher implements CustomFieldSearcher {

    private final FieldVisibilityManager fieldVisibilityManager;
    private final CustomFieldInputHelper customFieldInputHelper;
    private final JqlOperandResolver jqlOperandResolver;
    private final DoubleConverter doubleConverter;
    private final JiraAuthenticationContext authenticationContext;


    private volatile CustomFieldSearcherClauseHandler clauseHandler;
    private volatile CustomFieldSearcherInformation searcherInformation;
    private volatile SearchInputTransformer searchInputTransformer;
    private volatile SearchRenderer searchRenderer;

    public LocationSearcher(FieldVisibilityManager fieldVisibilityManager, CustomFieldInputHelper customFieldInputHelper,
                            JqlOperandResolver jqlOperandResolver, DoubleConverter doubleConverter,
                            JiraAuthenticationContext authenticationContext)
    {
        this.fieldVisibilityManager = fieldVisibilityManager;
        this.customFieldInputHelper = customFieldInputHelper;
        this.jqlOperandResolver = jqlOperandResolver;
        this.doubleConverter = doubleConverter;
        this.authenticationContext = authenticationContext;
    }


    public void init(CustomField field) {

        final FieldIndexer indexer = new LocationIndexer(fieldVisibilityManager, field, doubleConverter);
        final ClauseNames names = field.getClauseNames();
        final Set<Operator> supportedOps = EnumSet.of(Operator.LIKE, Operator.IN);
        this.searcherInformation = new CustomFieldSearcherInformation(field.getId(), field.getNameKey(),
                Collections.singletonList(indexer), new AtomicReference<CustomField>(field));
        this.searchRenderer = new CustomFieldRenderer(names, getDescriptor(), field,
                new SingleValueCustomFieldValueProvider(), fieldVisibilityManager);
        this.searchInputTransformer = new ExactTextCustomFieldSearchInputTransformer(field, names,
                searcherInformation.getId(), customFieldInputHelper);

        this.clauseHandler = new SimpleCustomFieldSearcherClauseHandler(
                new LocationSearchClauseValidator(supportedOps, jqlOperandResolver, authenticationContext.getI18nHelper()),
                new LocationLikeQueryFactory(jqlOperandResolver, field.getId(), doubleConverter, authenticationContext),
                supportedOps,
                JiraDataTypes.TEXT);
    }

    public CustomFieldSearcherClauseHandler getCustomFieldSearcherClauseHandler() {
        return checkInitialized(clauseHandler, "customFieldSearcherClauseHandler");
    }

    public SearcherInformation<CustomField> getSearchInformation() {
        return checkInitialized(searcherInformation, "searcherInformation");
    }

    public SearchInputTransformer getSearchInputTransformer() {
        return checkInitialized(searchInputTransformer, "searchInputTransformer");
    }

    public SearchRenderer getSearchRenderer() {
        return checkInitialized(searchRenderer, "searchRenderer");
    }

    private <T> T checkInitialized(T instance, String name) {
        checkState(instance != null, name + " not initialized! .init() has not been called yet");
        return instance;
    }
}
