package com.atlassian.jconnect.jira.customfields;

import com.atlassian.jira.issue.context.GlobalIssueContext;
import com.atlassian.jira.issue.context.JiraContextNode;
import com.atlassian.jira.issue.context.ProjectContext;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.fields.config.FieldConfigScheme;
import com.atlassian.jira.issue.fields.config.manager.FieldConfigSchemeManager;
import com.atlassian.jira.issue.fields.config.manager.IssueTypeSchemeManager;
import com.atlassian.jira.issue.fields.screen.FieldScreen;
import com.atlassian.jira.issue.fields.screen.FieldScreenManager;
import com.atlassian.jira.issue.fields.screen.FieldScreenScheme;
import com.atlassian.jira.issue.fields.screen.FieldScreenSchemeManager;
import com.atlassian.jira.issue.fields.screen.FieldScreenTab;
import com.atlassian.jira.issue.fields.screen.issuetype.IssueTypeScreenScheme;
import com.atlassian.jira.issue.fields.screen.issuetype.IssueTypeScreenSchemeEntity;
import com.atlassian.jira.issue.fields.screen.issuetype.IssueTypeScreenSchemeManager;
import com.atlassian.jira.issue.operation.IssueOperations;
import com.atlassian.jira.project.Project;
import com.google.common.collect.ImmutableList;

import java.util.Collection;
import java.util.List;

/**
 * Utility component to manage custom field schemes for JIRA Connect fields
 *
 */
public class CustomFieldSchemeHelper {

    private final CustomFieldHelper customFieldHelper;
    private final IssueTypeSchemeManager issueTypeManager;
    private final IssueTypeScreenSchemeManager issueTypeScreenSchemeManager;
    private final FieldConfigSchemeManager fieldConfigSchemeManager;
    private final FieldScreenManager fieldScreenManager;
    private final FieldScreenSchemeManager fieldScreenSchemeManager;


    public CustomFieldSchemeHelper(CustomFieldHelper customFieldHelper,
                                   IssueTypeSchemeManager issueTypeManager,
                                   FieldConfigSchemeManager fieldConfigSchemeManager,
                                   FieldScreenManager fieldScreenManager,
                                   FieldScreenSchemeManager fieldScreenSchemeManager,
                                   IssueTypeScreenSchemeManager issueTypeScreenSchemeManager) {
        this.customFieldHelper = customFieldHelper;
        this.issueTypeManager = issueTypeManager;
        this.fieldConfigSchemeManager = fieldConfigSchemeManager;
        this.fieldScreenManager = fieldScreenManager;
        this.fieldScreenSchemeManager = fieldScreenSchemeManager;
        this.issueTypeScreenSchemeManager = issueTypeScreenSchemeManager;
    }

    public void toggleFieldsToProjectScheme(boolean addFields, Project project, BuiltInField... fields) {

        final List<String> issueTypeIds = issueTypeIdsFor(project);
        for (BuiltInField field : fields) {
            CustomField customField = customFieldHelper.initBuiltInCustomField(field);
            
            if (addFields) {
                if (!customField.isInScope(project, issueTypeIds)) {
                    addFieldToScheme(project, field, customField);
                }
                if (field.isVisible()) // TODO: include opt-in variable for Location to be added to screens.
                {
                    addToScreens(customField, project);
                }
            } else {
                removeFieldFromSchemeAndScreens(project, field, customField);
            }

        }
        fieldScreenManager.refresh();
    }

    private void removeFieldFromSchemeAndScreens(Project project, BuiltInField field, CustomField customField) {
        final List<FieldConfigScheme> configSchemes = customField.getConfigurationSchemes();
        final ProjectContext projectContext = new ProjectContext(project.getId());
        for (FieldConfigScheme scheme : configSchemes) {
            final List<JiraContextNode> contextNodes = scheme.getContexts();
            if (contextNodes != null) {
                for (JiraContextNode contextNode : contextNodes) {
                    if (!(contextNode instanceof GlobalIssueContext) && contextNode.isInContext(projectContext)) {
                        fieldConfigSchemeManager.removeFieldConfigScheme(scheme.getId());
                        break;
                    }
                }
            }
        }
        fieldConfigSchemeManager.removeSchemeAssociation(projectContext(project), customField);
        fieldConfigSchemeManager.removeInvalidFieldConfigSchemesForCustomField(customField.getId());

    }

    private void addFieldToScheme(Project project, BuiltInField field, CustomField customField) {
        FieldConfigScheme scheme = fieldConfigSchemeManager.createDefaultScheme(customField, projectContext(project));
    }


    private void addToScreens(CustomField customField, Project project)
    {
        final IssueTypeScreenScheme screenScheme = this.issueTypeScreenSchemeManager.getIssueTypeScreenScheme(project.getGenericValue());

        Collection<IssueTypeScreenSchemeEntity> issueTypeScreenSchemeEntities = screenScheme.getEntities();
        for (IssueTypeScreenSchemeEntity schemeEntity : issueTypeScreenSchemeEntities)
        {
            final FieldScreenScheme fieldScreenScheme = schemeEntity.getFieldScreenScheme();

            final FieldScreen fieldScreen = fieldScreenScheme.getFieldScreen(IssueOperations.VIEW_ISSUE_OPERATION);
            if (fieldScreen != null && !fieldScreen.getTabs().isEmpty() && !fieldScreen.containsField(customField.getId()))
            {
                FieldScreenTab tab = fieldScreen.getTabs().iterator().next();
                tab.addFieldScreenLayoutItem(customField.getId());
            }
        }
    }

    private List<JiraContextNode> projectContext(Project project) {
        return ImmutableList.<JiraContextNode>of(new ProjectContext(project.getId()));
    }

    List<String> issueTypeIdsFor(Project project) {
        return CustomFieldHelper.issueTypeIdsFor(project, issueTypeManager);
    }

}
