package com.atlassian.jconnect.jira.customfields;

import com.atlassian.jira.issue.CustomFieldManager;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.fields.config.manager.IssueTypeSchemeManager;
import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.security.JiraAuthenticationContext;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.ofbiz.core.entity.GenericEntityException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;

import static com.google.common.collect.Lists.transform;

/**
 * Custom field logic for JIRA Connect.
 *
 */
public class CustomFieldHelper
{
    private static final Logger logger = LoggerFactory.getLogger(CustomFieldHelper.class);


    private final CustomFieldManager customFieldManager;
    private final IssueTypeSchemeManager issueTypeSchemeManager;
    private final JiraAuthenticationContext jiraAuthenticationContext;

    public CustomFieldHelper(CustomFieldManager customFieldManager, IssueTypeSchemeManager issueTypeSchemeManager,
                             JiraAuthenticationContext jiraAuthenticationContext)
    {
        this.customFieldManager = customFieldManager;
        this.issueTypeSchemeManager = issueTypeSchemeManager;
        this.jiraAuthenticationContext = jiraAuthenticationContext;
    }

    static List<String> issueTypeIdsFor(Project project, IssueTypeSchemeManager issueTypeSchemeManager) {
        return transform(ImmutableList.copyOf(issueTypeSchemeManager.getIssueTypesForProject(project)), new Function<IssueType, String>() {
            public String apply(@Nullable IssueType from) {
                return from.getId();
            }
        });
    }

    public boolean containsField(Project project, CustomField field)
    {
        if (field == null)
        {
            return false;
        }
        
        return field.isInScope(project, issueTypeIdsFor(project));
    }

    private List<String> issueTypeIdsFor(Project project) {
        return issueTypeIdsFor(project, issueTypeSchemeManager);
    }

    public boolean containsAllFields(Project project, CustomField... fields)
    {
        boolean result = true;
        for (CustomField field : fields)
        {
            result &= containsField(project, field);
        }
        return result;
    }

    public CustomField[] getAllBuiltInCustomFields() {
        return initBuiltInCustomFields(BuiltInField.values());
    }

    public CustomField[] initBuiltInCustomFields(BuiltInField... fields) {
        List<CustomField> collector = Lists.newArrayList();
        for (BuiltInField field : fields) {
            collector.add(initBuiltInCustomField(field));
        }
        return collector.toArray(new CustomField[collector.size()]);
    }

    public CustomField initBuiltInCustomField(BuiltInField field) {
        return initBuiltInCustomField(field, true);
    }

    private CustomField initBuiltInCustomField(BuiltInField fieldDescription, boolean createIfNecessary) {
        Collection<CustomField> fields = customFieldManager.getCustomFieldObjectsByName(fieldDescription.fieldName());
        if (fields != null)
        {
            for (CustomField field : fields)
            {
                if (field.getCustomFieldType().getKey().equals(fieldDescription.fieldTypeKey()))
                {
                    return field;
                }
            }
        }
        if (createIfNecessary) {
            return createBuiltInCustomField(fieldDescription);
        } else {
            logger.warn("Field not yet created: " + fieldDescription.fullDescription());
            return null;
        }
    }

    private CustomField createBuiltInCustomField(BuiltInField field) {
        try {
            return customFieldManager.createCustomField(
                                                            field.fieldName(),
                                                            getText("customfields.builtin.description"),
                                                            customFieldManager.getCustomFieldType(field.fieldTypeKey()),
                                                            customFieldManager.getCustomFieldSearcher(field.searcherKey()),
                                                            null,
                                                            null);
        } catch (GenericEntityException e) {
            logger.warn("Unable to create field " + field.fullDescription(), e);
            return null;
        }
    }

    private String getText(String key, Object... params) {
        if (params.length > 0) {
            return jiraAuthenticationContext.getI18nHelper().getText(key, params);
        } else {
            return jiraAuthenticationContext.getI18nHelper().getText(key);
        }
    }

}
