package com.atlassian.http.mime;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class can sniff a file according to the current settings specified by the MimeSNiffPolicyProvider and determine
 * how raw files should be handled.
 */
public class ContentDispositionHeaderGuesser {
    private static final Logger log = LoggerFactory.getLogger(ContentDispositionHeaderGuesser.class);

    private static final String CONTENT_DISPOSITION_ATTACHMENT = "attachment";
    private static final String CONTENT_DISPOSITION_INLINE = "inline";

    private final DownloadPolicyProvider downloadPolicyProvider;
    private final HostileExtensionDetector hostileExtensionDetector;

    public ContentDispositionHeaderGuesser(DownloadPolicyProvider downloadPolicyProvider, HostileExtensionDetector hostileExtensionDetector) {
        this.downloadPolicyProvider = downloadPolicyProvider;
        this.hostileExtensionDetector = hostileExtensionDetector;
    }

    /**
     * This will suggest a content disposition type (inline or attachment) for the given file, respecting the settings
     * given by the provided download policy.
     * <p/>
     * Special treatment is given to html/xml - they will always be displayed inline if the contentType is set to
     * a text type, and if the browser is not IE.
     *
     * @param fileName the name of the file
     * @param mimeContentType the exisiting content type
     * @return either INLINE or ATTACHMENT ready for a Content-Disposition header
     */
    public String guessContentDispositionHeader(String fileName, String mimeContentType, String userAgent) {
        final DownloadPolicy downloadPolicy = downloadPolicyProvider.getPolicy();
        boolean forceDownload = false;
        if (downloadPolicy != DownloadPolicy.Insecure && isExecutableContent(fileName, mimeContentType)) {
            // only in insecure mode we allow direct view of executable content
            forceDownload = true;
            if (log.isDebugEnabled()) {
                log.debug("\"" + fileName + "\" (" + mimeContentType + ")" + " presents as executable content, forcing download.");
            }
        } else if (downloadPolicy == DownloadPolicy.Secure) {
            forceDownload = true;
        }

        //allow inline view for browsers except IE, but only in smart mode
        if (forceDownload
                && isAllowInlineOverride(fileName, mimeContentType, userAgent, downloadPolicy)) {
            forceDownload = false;
        }

        return forceDownload ? CONTENT_DISPOSITION_ATTACHMENT : CONTENT_DISPOSITION_INLINE;
    }

    /**
     * Given a filename, its MIME type, and a browser user agent, return a 'safe' mime type, depending on
     * the download policy configured, such that 'safe' browsers can display the executable text file without causing xss issues.
     * <p/>
     * If the download policy is set to {@link DownloadPolicy#Secure} or {@link DownloadPolicy#Insecure}, then
     * the given mimeContentType will be returned.
     * <p/>
     * If the download policy is set to {@link DownloadPolicy#Smart}, and the user agent is not Internet Explorer
     * , then for text files that can be displayed in  plain text (e.g., text/html, as determined by
     * {@link HostileExtensionDetector#isTextContentType(String)} and {@link HostileExtensionDetector#isTextExtension(String)}),
     * a 'text/plain' mime type will be returned instead of the given mimeContentType.
     *
     * @param filename the file to display
     * @param mimeContentType the mime type for the given file
     * @param userAgent the browser user agent.
     * @return a mime type
     */
    public String guessMIME(String filename, String mimeContentType, String userAgent) {
        if (CONTENT_DISPOSITION_INLINE.equals(guessContentDispositionHeader(filename, mimeContentType, userAgent))
                && isAllowInlineOverride(filename, mimeContentType, userAgent, downloadPolicyProvider.getPolicy())) {
            return "text/plain";
        } else {
            return mimeContentType;
        }
    }

    private boolean isAllowInlineOverride(String fileName, String mimeContentType, String userAgent, DownloadPolicy downloadPolicy) {
        return downloadPolicy == DownloadPolicy.Smart
                && !BrowserUtils.isIE(userAgent)
                && isTextContent(fileName, mimeContentType);
    }

    private boolean isTextContent(String fileName, String mimeContentType) {
        return hostileExtensionDetector.isTextContent(fileName, mimeContentType);
    }

    private boolean isExecutableContent(String name, String mimeContentType) {
        return hostileExtensionDetector.isExecutableContent(name, mimeContentType);
    }
}