package com.atlassian.gadgets.dashboard;

import com.atlassian.gadgets.DashboardItemState;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import javax.annotation.concurrent.Immutable;
import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static java.util.Collections.nCopies;

@Immutable
public final class DashboardColumns implements Serializable {
    private final List<List<DashboardItemState>> columns;

    private DashboardColumns(final List<List<DashboardItemState>> columns) {
        this.columns = columns;
    }

    /**
     * Returns an immutable {@code List} of the {@code DashboardStateItem}s in the given {@code column}. <br />
     * They are returned in the order that they appear on the dashboard from top to bottom.
     *
     * @param column the index of the column to retrieve the {@code DashboardItemState}s for
     * @return an immutable {@code List} of the {@code DashboardItemState}s in the {@code column}
     */
    public List<DashboardItemState> getItemsInColumn(DashboardState.ColumnIndex column) {
        return columns.get(column.index());
    }

    /**
     * Returns an immutable List of all columns represented as a list of items. <br />
     * Note that all those lists are immutable.
     */
    public List<List<DashboardItemState>> getColumns() {
        return columns;
    }

    public int numberOfColumns() {
        return columns.size();
    }

    public static DashboardColumns from(Iterable<? extends Iterable<DashboardItemState>> columns, int numberOfColumns) {
        return new DashboardColumns(copy(columns, numberOfColumns));
    }

    private static List<List<DashboardItemState>> copy(Iterable<? extends Iterable<DashboardItemState>> columns, int size) {
        List<List<DashboardItemState>> listBuilder = Lists.newArrayList();
        if (columns != null) {
            for (Iterable<DashboardItemState> column : columns) {
                listBuilder.add(ImmutableList.copyOf(column));
            }
        }
        pad(listBuilder, size, Collections.<DashboardItemState>emptyList());
        return ImmutableList.copyOf(listBuilder);
    }

    private static <E, T extends E> void pad(Collection<E> list, int toSize, T obj) {
        list.addAll(nCopies(toSize - list.size(), obj));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        DashboardColumns that = (DashboardColumns) o;

        return Objects.equal(this.columns, that.columns);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(columns);
    }

    @Override
    public String toString() {
        return columns.toString();
    }
}
