package com.atlassian.diagnostics.internal.platform.plugin;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.event.events.PluginDisabledEvent;
import com.atlassian.plugin.event.events.PluginEnabledEvent;
import com.atlassian.plugin.event.events.PluginFrameworkStartedEvent;
import com.atlassian.plugin.osgi.container.OsgiContainerManager;
import com.atlassian.plugin.osgi.util.OsgiHeaderUtil;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.wiring.BundleWiring;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.AbstractMap;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.Supplier;
import java.util.stream.Stream;

import static java.util.Collections.emptyMap;
import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static java.util.concurrent.TimeUnit.SECONDS;
import static java.util.stream.Collectors.toConcurrentMap;
import static java.util.stream.Collectors.toMap;
import static org.osgi.framework.Bundle.ACTIVE;
import static org.osgi.framework.wiring.BundleWiring.LISTRESOURCES_RECURSE;

/**
 * A store of mappings from canonical classnames to the plugin likely to be the invoker.
 *
 * @since 1.2.0
 */
@ParametersAreNonnullByDefault
@ReturnValuesAreNonnullByDefault
public class ClassNameToPluginKeyStore {
    private static final Logger log = LoggerFactory.getLogger(ClassNameToPluginKeyStore.class);
    @VisibleForTesting
    static final String CLASS_FILE_EXTENSION = ".class";
    private final BundleSupplier bundleSupplier;
    // Fetching the reference from main memory instead of the CPU cache adds in the order of tens of nanoseconds, which
    // is insignificant compared to the whole monitoring chain. This is worth it for consistent results.
    private volatile static Map<String, String> classNameToPluginKeyMap = emptyMap();
    private final PluginSystemMonitoringConfig pluginSystemMonitoringConfig;
    private static Timer delayMapGenerationTimer = newTimer();

    public ClassNameToPluginKeyStore(
            final EventPublisher eventPublisher,
            final OsgiContainerManager osgiContainerManager,
            final PluginSystemMonitoringConfig pluginSystemMonitoringConfig
    ) {
        this(osgiContainerManager::getBundles, eventPublisher, pluginSystemMonitoringConfig);
    }

    public ClassNameToPluginKeyStore(
            final BundleContext bundleContext,
            final EventPublisher eventPublisher,
            final PluginSystemMonitoringConfig pluginSystemMonitoringConfig
    ) {
        this(bundleContext::getBundles, eventPublisher, pluginSystemMonitoringConfig);
    }

    private ClassNameToPluginKeyStore(
            final BundleSupplier bundleSupplier,
            final EventPublisher eventPublisher,
            final PluginSystemMonitoringConfig pluginSystemMonitoringConfig
    ) {
        this.bundleSupplier = bundleSupplier;
        this.pluginSystemMonitoringConfig = pluginSystemMonitoringConfig;
        eventPublisher.register(this);
    }

    /**
     * Will try to get a unique plugin key from a classname. This might not be possible if the classname is present in
     * multiple plugin bundles.
     *
     * @param classname Canonical classname e.g {@code com.atlassian.jira.ExampleClass.InnerClass}
     * @return the key of the sole plugin that is associated with the classname. Empty if none exists or associated with
     * multiple plugins
     */
    public Optional<String> getPluginKey(@Nullable final String classname) {
        if (classname == null) {
            return Optional.empty();
        }

        return Optional.ofNullable(classNameToPluginKeyMap.get(classname));
    }

    @EventListener
    public void onPluginFrameworkStartedEvent(final PluginFrameworkStartedEvent event) {
        scheduleNewMapGeneration(event);
    }

    @EventListener
    public void onPluginEnabled(final PluginEnabledEvent event) {
        scheduleNewMapGeneration(event);
    }

    @EventListener
    public void onPluginDisabled(final PluginDisabledEvent event) {
        scheduleNewMapGeneration(event);
    }

    private void scheduleNewMapGeneration(final Object triggeringEvent) {
        try {
            classNameToPluginKeyMap = emptyMap(); // Invalidate immediately since results may no longer be accurate

            delayMapGenerationTimer.cancel();
            delayMapGenerationTimer = newTimer();
            delayMapGenerationTimer.schedule(newMapRefreshTask(triggeringEvent), SECONDS.toMillis(30L));
        } catch (final Exception exception) { // Don't want to blow-up when starting the plugin system
            log.debug("Failed to schedule task to generate a map of class names to plugin keys", exception);
        }
    }

    @VisibleForTesting
    TimerTask newMapRefreshTask(final Object triggeringEvent) {
        return new TimerTask() {
            @Override
            public void run() {
                log.debug("Refreshing classname to plugin key map after: {}", triggeringEvent);
                classNameToPluginKeyMap = generateClassNameToPluginKeyMap();
            }
        };
    }

    @VisibleForTesting
    private Map<String, String> generateClassNameToPluginKeyMap() {
        if (isNull(bundleSupplier) || pluginSystemMonitoringConfig.classNameToPluginKeyStoreDisabled()) {
            return emptyMap(); // Should clear if disabled at runtime
        }

        return Arrays.stream(bundleSupplier.get())
                .filter(Objects::nonNull)
                .filter(ClassNameToPluginKeyStore::pluginIsEnabled)
                .map(ClassNameToPluginKeyStore::getBundleWiring)
                .filter(Objects::nonNull)
                .flatMap(ClassNameToPluginKeyStore::classNameToPluginKeyFanOut)
                // if there's anything other than 1 plugin key then we can't attribute anything anyway
                .collect(toMap(Map.Entry::getKey, Map.Entry::getValue, (first, second) -> null))
                // save some memory
                .entrySet()
                .stream()
                .filter(entry -> nonNull(entry.getValue()))
                .collect(toConcurrentMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private static boolean pluginIsEnabled(final Bundle bundle) {
        return bundle.getState() == ACTIVE;
    }

    private static BundleWiring getBundleWiring(final Bundle bundle) {
        return bundle.adapt(BundleWiring.class);
    }

    private static Stream<Map.Entry<String, String>> classNameToPluginKeyFanOut(final BundleWiring bundleWiring) {
        final String pluginKey = OsgiHeaderUtil.getPluginKey(bundleWiring.getBundle());
        return listAllClasses(bundleWiring).stream()
                .map(ClassNameToPluginKeyStore::resourceFilePathToCanonicalClassName)
                .map(className -> new AbstractMap.SimpleEntry<>(className, pluginKey));
    }

    private static Collection<String> listAllClasses(final BundleWiring bundleWiring) {
        return bundleWiring.listResources("/", "*" + CLASS_FILE_EXTENSION, LISTRESOURCES_RECURSE);
    }

    @VisibleForTesting
    static String resourceFilePathToCanonicalClassName(final String resourceName) {
        return resourceName.substring(0, resourceName.length() - CLASS_FILE_EXTENSION.length()).replaceAll("/", ".");
    }

    private static Timer newTimer() {
        // Using a daemon thread means the JVM doesn't have to wait for it to complete before shutting down
        return new Timer("generate-classname-to-pluginKey-map", true);
    }

    private interface BundleSupplier extends Supplier<Bundle[]> {
    }
}
