package com.atlassian.diagnostics.internal.platform.monitor.scheduler;

import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.SchedulerHistoryService;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.caesium.impl.CaesiumSchedulerService;
import com.atlassian.scheduler.caesium.spi.CaesiumSchedulerConfiguration;
import com.atlassian.scheduler.config.JobId;
import com.atlassian.scheduler.config.JobRunnerKey;
import com.atlassian.scheduler.core.SchedulerServiceController;
import com.atlassian.scheduler.status.JobDetails;
import com.atlassian.scheduler.status.RunDetails;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

public class DefaultSchedulerDiagnosticProvider implements SchedulerDiagnosticProvider {

    private final CaesiumSchedulerConfiguration caesiumSchedulerConfiguration;
    private final SchedulerHistoryService schedulerHistoryService;
    private final SchedulerServiceController schedulerServiceController;
    private final SchedulerService schedulerService;

    private Optional<CaesiumSchedulerService> caesiumSchedulerService = Optional.empty();

    public DefaultSchedulerDiagnosticProvider(final CaesiumSchedulerConfiguration caesiumSchedulerConfiguration,
                                              final SchedulerHistoryService schedulerHistoryService,
                                              final SchedulerServiceController schedulerServiceController,
                                              final SchedulerService schedulerService) {
        this.caesiumSchedulerConfiguration = caesiumSchedulerConfiguration;
        this.schedulerHistoryService = schedulerHistoryService;
        this.schedulerServiceController = schedulerServiceController;
        this.schedulerService = schedulerService;
        if (schedulerService instanceof CaesiumSchedulerService) {
            this.caesiumSchedulerService = Optional.of((CaesiumSchedulerService) schedulerService);
        } else if (schedulerServiceController instanceof CaesiumSchedulerService) {
            this.caesiumSchedulerService = Optional.of((CaesiumSchedulerService) schedulerServiceController);
        }
    }

    @Override
    public SchedulerDiagnostic getDiagnostic() {
        final int workerThreadCount = caesiumSchedulerConfiguration.workerThreadCount();
        return new SchedulerDiagnostic(workerThreadCount, getRunningJobs(), getScheduledJobs());
    }

    private List<RunningJobDiagnostic> getRunningJobs() {
        return schedulerServiceController.getLocallyRunningJobs().stream()
                .map(runningJob -> {
                    final Optional<JobRunner> jobRunner = getJobRunner(runningJob.getJobConfig().getJobRunnerKey());
                    return new RunningJobDiagnostic(runningJob, jobRunner);
                }).collect(Collectors.toList());
    }

    private List<ScheduledJobDiagnostic> getScheduledJobs() {
        final List<JobDetails> allScheduledJobsJobDetails = schedulerService.getJobsByJobRunnerKeys(new ArrayList<>(schedulerService.getJobRunnerKeysForAllScheduledJobs()));
        final Map<JobId, RunDetails> runDetailsByJobId = schedulerHistoryService.getLastRunForJobs(allScheduledJobsJobDetails.stream().map(JobDetails::getJobId).collect(Collectors.toList()));

        return allScheduledJobsJobDetails.stream()
                    .map(jobDetails -> new ScheduledJobDiagnostic(
                            jobDetails,
                            getRunDetails(jobDetails, runDetailsByJobId),
                            getJobRunner(jobDetails.getJobRunnerKey())
                    ))
                .collect(Collectors.toList());
    }

    private Optional<RunDetails> getRunDetails(final JobDetails jobDetails, final Map<JobId, RunDetails> jobDetailsRunDetailsMap) {
        return Optional.ofNullable(jobDetailsRunDetailsMap.get(jobDetails.getJobId()));
    }

    private Optional<JobRunner> getJobRunner(final JobRunnerKey jobRunnerKey) {
        return caesiumSchedulerService.map(service -> service.getJobRunner(jobRunnerKey));
    }
}
