package com.atlassian.diagnostics.internal.platform;


import com.atlassian.diagnostics.internal.platform.poller.DiagnosticPoller;

import javax.annotation.concurrent.NotThreadSafe;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.function.Supplier;

import static com.atlassian.diagnostics.internal.platform.monitor.DurationUtils.durationOf;

/**
 * This class is used to keep track of consecutive alerts that reach a time limit.
 * It is designed to be ran on a single @{@link DiagnosticPoller} thread.
 */
@NotThreadSafe
public class ConsecutiveAlertGate {

    private final Duration timeLimit;
    private final Clock clock;
    private Instant lastExecution;

    ConsecutiveAlertGate(final Supplier<Duration> consecutiveAlertDurationThreshold, final Clock clock) {
        this.timeLimit = consecutiveAlertDurationThreshold.get();
        this.clock = clock;
        this.lastExecution = clock.instant();
    }

    public boolean shouldRaiseAlert(final DiagnosticAlert alert) {
        if (alert.isRaised() && hasAlertTimeLimitBeenReached()) {
            lastExecution = clock.instant();
            return true;
        }

        return false;
    }

    private boolean hasAlertTimeLimitBeenReached() {
        return durationOf(Duration.between(lastExecution, clock.instant())).isGreaterThanOrEqualTo(timeLimit);
    }

}
