package com.atlassian.diagnostics.internal.platform.plugin;

import com.atlassian.annotations.VisibleForTesting;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import static java.lang.Math.max;
import static java.time.Duration.ofHours;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;

public class PluginFinderImpl implements PluginFinder {
    private static final Logger log = LoggerFactory.getLogger(PluginFinderImpl.class);
    @VisibleForTesting
    static final String JAVA_CLASS_PREFIX = "java";

    private final ClassContextSecurityManager securityManger;
    private final BundleFinder bundleFinder;
    private final ClassNameToPluginKeyStore classNameToPluginKeyStore;
    private final Cache<Class<?>, String> classPluginSourceCache;
    private final int classContextTraversalLimit;
    private final int stackTraceTraversalLimit;

    public PluginFinderImpl(
            final BundleFinder bundleFinder,
            final ClassNameToPluginKeyStore classNameToPluginKeyStore,
            final PluginSystemMonitoringConfig pluginSystemMonitoringConfig
    ) {
        this(createClassContextSecurityManager(),
                bundleFinder,
                classNameToPluginKeyStore,
                CacheBuilder.newBuilder()
                        .maximumSize(10_000)
                        .weakValues()
                        .expireAfterAccess(ofHours(1))
                        .build(),
                pluginSystemMonitoringConfig.classContextTraversalLimit(),
                pluginSystemMonitoringConfig.stackTraceTraversalLimit()
        );
    }

    @VisibleForTesting
    PluginFinderImpl(
            final ClassContextSecurityManager securityManger,
            final BundleFinder bundleFinder,
            final ClassNameToPluginKeyStore classNameToPluginKeyStore,
            final Cache<Class<?>, String> classPluginSourceCache,
            final int classContextTraversalLimit,
            final int stackTraceTraversalLimit
    ) {
        this.securityManger = securityManger;
        this.bundleFinder = bundleFinder;
        this.classNameToPluginKeyStore = classNameToPluginKeyStore;
        this.classPluginSourceCache = classPluginSourceCache;
        this.classContextTraversalLimit = classContextTraversalLimit;
        this.stackTraceTraversalLimit = stackTraceTraversalLimit;
    }

    private static ClassContextSecurityManager createClassContextSecurityManager() {
        try {
            return new ClassContextSecurityManager();
        } catch (final Exception exception) {
            log.debug("Failed to create security manager", exception);
            return null;
        }
    }

    static class ClassContextSecurityManager extends SecurityManager {
        private static final Class<?>[] EMPTY_ARRAY = new Class<?>[0];

        @Override
        protected Class<?>[] getClassContext() {
            final Class<?>[] classContext = super.getClassContext();
            return (classContext == null) ? EMPTY_ARRAY : classContext;
        }
    }

    public Collection<String> getPluginNamesInCurrentCallStack() {
        try {
            if (securityManger != null) {
                return getPluginsFromClasses(securityManger.getClassContext());
            }
        } catch (final Exception exception) {
            log.debug("Failed to get plugins list from call stack", exception);
        }
        return emptyList();
    }

    @Override
    public Collection<String> getPluginNamesFromStackTrace(@Nonnull final StackTraceElement[] stackTrace) {
        return Arrays.stream(stackTrace)
                .map(StackTraceElement::getClassName)
                .map(PluginFinderImpl::getClassNameWithoutLambda)
                .map(classNameToPluginKeyStore::getPluginKey)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(toSet());
    }

    @Nullable
    public String getInvokingPluginKeyFromStackTrace(final StackTraceElement[] stackTrace) {
        if (stackTrace == null) {
            return null;
        }

        final int traversalDepth = max(0, stackTrace.length - stackTraceTraversalLimit);
        for (int traceIndex = stackTrace.length - 1; traceIndex >= traversalDepth; traceIndex--) {
            if (!stackTrace[traceIndex].getClassName().startsWith(JAVA_CLASS_PREFIX)) {
                return classNameToPluginKeyStore
                        .getPluginKey(getClassNameWithoutLambda(stackTrace[traceIndex].getClassName()))
                        .orElse(null);
            }
        }

        return null;
    }

    private static String getClassNameWithoutLambda(final String clasName) {
        final int positionOfLambda = clasName.indexOf("$");
        return positionOfLambda < 0 ? clasName : clasName.substring(0, positionOfLambda);
    }

    @Nullable
    public String getInvokingPluginKeyFromClassContext(final Class<?>[] classContext) {
        if (classContext == null) {
            return null;
        }

        final int traversalDepth = max(0, classContext.length - classContextTraversalLimit);
        for (int traceIndex = classContext.length - 1; traceIndex >= traversalDepth; traceIndex--) {
            final Optional<String> pluginKey = bundleFinder.getBundleNameForClass(classContext[traceIndex]);

            if (pluginKey.isPresent()) {
                return pluginKey.get();
            }
        }

        return null;
    }

    private Collection<String> getPluginsFromClasses(final Class<?>[] classes) {
        final Set<String> plugins = new HashSet<>();
        for (final Class<?> clazz : classes) {
            final String cachedPluginSource = classPluginSourceCache.getIfPresent(clazz);

            if (cachedPluginSource == null) {
                resolvePlugin(clazz).ifPresent(pluginName -> add(plugins, pluginName));
            } else if (isNotEmpty(cachedPluginSource)) {
                add(plugins, cachedPluginSource);
            }
        }

        return plugins;
    }

    private Optional<String> resolvePlugin(final Class<?> clazz) {
        final Optional<String> pluginName = bundleFinder.getBundleNameForClass(clazz);
        classPluginSourceCache.put(clazz, pluginName.orElse(""));

        return pluginName;
    }

    private void add(final Set<String> plugins, final String pluginName) {
        plugins.add(pluginName);
    }
}
