package com.atlassian.diagnostics.internal.platform.monitor.operatingsystem.cpu;

import com.atlassian.diagnostics.internal.platform.monitor.operatingsystem.OperatingSystemMonitor;
import com.atlassian.diagnostics.internal.platform.monitor.operatingsystem.OperatingSystemMonitorConfiguration;
import com.atlassian.diagnostics.internal.platform.poller.DiagnosticPoller;

import java.time.Duration;
import java.time.Instant;

public class CPUPerformancePoller extends DiagnosticPoller<OperatingSystemMonitorConfiguration> {

    private final OperatingSystemMonitor operatingSystemMonitor;
    private final CPUDiagnosticProvider cpuDiagnosticProvider;
    private Duration consecutiveHighCpuUsageTime = Duration.ofMinutes(0);

    public CPUPerformancePoller(final OperatingSystemMonitorConfiguration config,
                                final OperatingSystemMonitor operatingSystemMonitor,
                                final CPUDiagnosticProvider cpuDiagnosticProvider) {
        super(CPUPerformancePoller.class.getName(), config);
        this.operatingSystemMonitor = operatingSystemMonitor;
        this.cpuDiagnosticProvider = cpuDiagnosticProvider;
    }

    @Override
    protected void execute() {
        final double systemCpuLoad = cpuDiagnosticProvider.getDiagnostics().getSystemCpuLoad();

        updateConsecutiveHighCpuUsageTime(systemCpuLoad, monitorConfiguration.getCpuUsagePercentageThreshold());

        if (consecutiveHighCpuUsageTime.toMinutes() >= monitorConfiguration.getMaximumHighCpuUsageTime().toMinutes()) {
            operatingSystemMonitor.raiseAlertForHighCpu(Instant.now(), new HighCPUUsageEvent(systemCpuLoad));
        }
    }

    private void updateConsecutiveHighCpuUsageTime(final double systemCpuLoad, final double cpuUsageThreshold) {
        if (systemCpuLoad > cpuUsageThreshold) {
            consecutiveHighCpuUsageTime = Duration.ofMinutes(consecutiveHighCpuUsageTime.toMinutes() + 1);
        } else {
            consecutiveHighCpuUsageTime = Duration.ofMinutes(0);
        }
    }
}
