package com.atlassian.diagnostics.internal.platform.monitor.db;

import com.atlassian.diagnostics.internal.jmx.JmxService;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.util.EnumSet;
import java.util.List;

public class DatabasePoolDiagnosticProvider {

    private static final int IDLE_CONNECTION_ATTRIBUTE_INDEX = 0;
    private static final int ACTIVE_CONNECTION_ATTRIBUTE_INDEX = 1;
    private static final int MAX_CONNECTION_ATTRIBUTE_INDEX = 2;

    private final JmxService jmxService;
    private JmxDatabasePoolAttributes jmxDatabasePoolAttributes = JmxDatabasePoolAttributes.UNKNOWN;

    public DatabasePoolDiagnosticProvider(final JmxService jmxService) {
        this.jmxService = jmxService;
        resolveJmxDatabasePoolAttributes();
    }

    @Nonnull
    public DatabasePoolDiagnostic getDiagnostic() {
        resolveJmxDatabasePoolAttributes();
        final List<Integer> databasePoolAttributeValues = jmxService.getJmxAttributes(
                jmxDatabasePoolAttributes.instanceOfQuery,
                new String[]{
                        jmxDatabasePoolAttributes.idleConnectionsAttributeName,
                        jmxDatabasePoolAttributes.activeConnectionsAttribute,
                        jmxDatabasePoolAttributes.maxConnectionsAttribute
                }
        );

        if (databasePoolAttributeValues.size() == 3) {
            return new DatabasePoolDiagnostic(
                    databasePoolAttributeValues.get(IDLE_CONNECTION_ATTRIBUTE_INDEX),
                    databasePoolAttributeValues.get(ACTIVE_CONNECTION_ATTRIBUTE_INDEX),
                    databasePoolAttributeValues.get(MAX_CONNECTION_ATTRIBUTE_INDEX)
            );
        }

        return DatabasePoolDiagnostic.EMPTY;
    }

    @Nonnull
    public Duration getPoolConnectionLeakTimeout() {
        resolveJmxDatabasePoolAttributes();
        final Integer abandonedTimeoutValue = jmxService.getJmxAttribute(
                jmxDatabasePoolAttributes.instanceOfQuery,
                jmxDatabasePoolAttributes.abandonedTimeoutAttributeName
        );

        if (abandonedTimeoutValue != null && abandonedTimeoutValue != Integer.MAX_VALUE) {
            return Duration.ofSeconds(abandonedTimeoutValue);
        } else {
            return Duration.ZERO;
        }
    }

    private void resolveJmxDatabasePoolAttributes() {
        if (jmxDatabasePoolAttributes.equals(JmxDatabasePoolAttributes.UNKNOWN)) {
            jmxDatabasePoolAttributes = EnumSet.allOf(JmxDatabasePoolAttributes.class).stream()
                    .filter(jmxDatabasePoolAttributes -> jmxService.hasObjectName(jmxDatabasePoolAttributes.instanceOfQuery))
                    .findAny()
                    .orElse(JmxDatabasePoolAttributes.UNKNOWN);
        }
    }
}
