package com.atlassian.diagnostics.internal.platform.monitor.cluster;

import com.atlassian.beehive.core.ManagedClusterLock;
import com.atlassian.beehive.core.ManagedClusterLockService;
import com.atlassian.beehive.core.stats.StatisticsKey;
import com.atlassian.diagnostics.DiagnosticsConfiguration;
import com.atlassian.diagnostics.internal.platform.poller.DiagnosticPoller;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class ClusterLockPoller extends DiagnosticPoller<ClusterLockMonitorConfiguration> {

    private final ClusterLockMonitor clusterLockMonitor;
    private final ManagedClusterLockService clusterLockService;
    private final DiagnosticsConfiguration diagnosticsConfiguration;

    public ClusterLockPoller(final ClusterLockMonitor clusterLockMonitor,
                             final ManagedClusterLockService clusterLockService,
                             final ClusterLockMonitorConfiguration clusterLockMonitorConfiguration,
                             final DiagnosticsConfiguration diagnosticsConfiguration) {
        super(ClusterLockPoller.class.getName(), clusterLockMonitorConfiguration);
        this.clusterLockMonitor = clusterLockMonitor;
        this.clusterLockService = clusterLockService;
        this.diagnosticsConfiguration = diagnosticsConfiguration;
    }

    @Override
    protected void execute() {
        final Collection<ManagedClusterLock> locks = clusterLockService.getAllKnownClusterLocks().stream()
                .filter(isLockedOnThisNode())
                .collect(Collectors.toList());

        raiseAlertIfLongWaitForLock(locks);
        raiseAlertIfQueueSizeIsHigh(locks);
    }

    private Predicate<ManagedClusterLock> isLockedOnThisNode() {
        return managedClusterLock -> diagnosticsConfiguration.getNodeName().equals(managedClusterLock.getClusterLockStatus().getLockedByNode())
                && managedClusterLock.isLocked();
    }

    private void raiseAlertIfLongWaitForLock(final Collection<ManagedClusterLock> clusterLocks) {
        final List<ManagedClusterLock> locksWithLongHoldTime = clusterLocks.stream()
                .filter(isWaitTimeAboveThreshold())
                .collect(Collectors.toList());

        if (!locksWithLongHoldTime.isEmpty()) {
            clusterLockMonitor.raiseAlertForLocksWithLongWait(Instant.now(), locksWithLongHoldTime);
        }
    }

    private Predicate<ManagedClusterLock> isWaitTimeAboveThreshold() {
        return managedClusterLock -> {
            final Long averageWaitTimeForLock = managedClusterLock.getStatistics().getOrDefault(StatisticsKey.AVERAGE_WAIT_TIME_MILLIS, 0L);
            final Long queueSize = managedClusterLock.getStatistics().getOrDefault(StatisticsKey.WAIT_QUEUE_LENGTH, 0L);

            return (averageWaitTimeForLock * queueSize) >= monitorConfiguration.lockWaitTimeThreshold();
        };
    }

    private void raiseAlertIfQueueSizeIsHigh(final Collection<? extends ManagedClusterLock> clusterLocks) {
        final List<ManagedClusterLock> locksWithLongWaits = clusterLocks.stream()
                .filter(isQueueSizeAboveThreshold())
                .collect(Collectors.toList());

        if (!locksWithLongWaits.isEmpty()) {
            clusterLockMonitor.raiseAlertForLargeQueue(Instant.now(), locksWithLongWaits);
        }
    }

    private Predicate<ManagedClusterLock> isQueueSizeAboveThreshold() {
        return clusterLock -> clusterLock.getStatistics().getOrDefault(StatisticsKey.WAIT_QUEUE_LENGTH, 0L) >= monitorConfiguration.queueSizeThreshold();
    }
}
