package com.atlassian.diagnostics.internal.platform.monitor.event;

import com.atlassian.diagnostics.AlertTrigger;
import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.detail.ThreadDumpProducer;
import com.atlassian.diagnostics.internal.InitializingMonitor;
import com.atlassian.diagnostics.internal.concurrent.Gate;
import com.atlassian.event.spi.ListenerInvoker;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;

import javax.annotation.Nonnull;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Set;
import java.util.stream.Collectors;

import static com.atlassian.diagnostics.internal.platform.monitor.DurationUtils.durationOf;
import static java.util.Objects.requireNonNull;

/**
 * Monitor for the event system that encapsulates all interactions with the atlassian-diagnostics framework
 * {@link EventSystemMonitor#init} must be called to begin operations
 */
public class EventSystemMonitor extends InitializingMonitor {

    private static final String KEY_PREFIX = "diagnostics.event.issue";

    static final int ID_EVENT_DROPPED = 1001;
    static final int ID_SLOW_LISTENER = 2001;

    private final EventSystemMonitorConfig config;
    private final Object lock;
    private final Set<Thread> invokerThreads;
    private final ThreadDumpProducer threadDumpProducer;
    private final Gate threadDumpGate;
    private final Clock clock;

    EventSystemMonitor(@Nonnull final Clock clock,
                       @Nonnull final EventSystemMonitorConfig config,
                       @Nonnull final ThreadDumpProducer threadDumpProducer) {
        this.clock = requireNonNull(clock, "clock");
        this.config = requireNonNull(config, "config");
        this.threadDumpGate = new Gate(clock, config.getEventDroppedAlertThreadDumpCoolDown());
        this.threadDumpProducer = requireNonNull(threadDumpProducer, "threadDumpProducer");

        lock = new Object();
        invokerThreads = Sets.newConcurrentHashSet();
    }

    public EventSystemMonitor(@Nonnull final EventSystemMonitorConfig config,
                              @Nonnull final ThreadDumpProducer threadDumpProducer) {
        this(Clock.systemDefaultZone(), config, threadDumpProducer);
    }

    @Override
    public void init(final MonitoringService monitoringService) {
        synchronized (lock) {
            monitor = monitoringService.createMonitor("EVENT", "diagnostics.event.name");
            defineIssues();
        }
    }

    protected void defineIssues() {
        defineIssue(KEY_PREFIX, ID_EVENT_DROPPED, Severity.ERROR, EventDroppedDetails.class);
        defineIssue(KEY_PREFIX, ID_SLOW_LISTENER, Severity.WARNING);
    }

    /**
     * Raises an alert that an event has been dropped
     *  @param timestamp the timestamp to be used on the alert
     * @param queueLength the length of the (full) event queue
     * @param eventClass the type of event that was dropped
     */
    void alertEventDropped(@Nonnull final Instant timestamp, final int queueLength, @Nonnull final Class<?> eventClass) {
        requireNonNull(timestamp, "timestamp");

        alert(ID_EVENT_DROPPED, builder -> builder
                .timestamp(timestamp)
                .details(() -> {
                    EventDroppedDetails.Builder detailsBuilder =
                            new EventDroppedDetails.Builder(eventClass.getName(), queueLength);

                    threadDumpGate.ifAccessible(() -> {
                        detailsBuilder.threadDumps(threadDumpProducer.produce(getEventPoolThreads()));
                        return null;
                    });
                    return detailsBuilder.build();
                }));
    }

    /**
     * Invokes the listener and monitors the invocation
     *
     * @param trigger the trigger for the event
     * @param delegate the delete for invocation
     * @param event the event
     */
    void invokeMonitored(@Nonnull final AlertTrigger trigger, @Nonnull final ListenerInvoker delegate, @Nonnull final Object event) {
        Instant start = clock.instant();
        invokerThreads.add(Thread.currentThread());
        try {
            delegate.invoke(event);
        } finally {
            invokerThreads.remove(Thread.currentThread());
            Duration duration = Duration.between(start, clock.instant());
            if (durationOf(duration).isGreaterThanOrEqualTo(config.getSlowListenerAlertDuration(trigger))) {
                alert(ID_SLOW_LISTENER, builder -> builder
                        .timestamp(start)
                        .trigger(trigger)
                        .details(() -> ImmutableMap.of(
                                "timeMillis", duration.toMillis(),
                                "eventType", event.getClass().getName())));
            }
        }
    }

    private Set<Thread> getEventPoolThreads() {
        // only return threads that are part of the event dispatcher thread pool
        return config.getEventThreadGroup()
                .map(threadGroup ->
                        invokerThreads.stream()
                        .filter(thread -> thread.getThreadGroup().equals(threadGroup))
                        .collect(Collectors.toSet()))
                .orElse(invokerThreads);
    }
}
