package com.atlassian.diagnostics.internal.platform.monitor.db;

import com.atlassian.diagnostics.internal.platform.ConsecutiveAlertGate;
import com.atlassian.diagnostics.internal.platform.ConsecutiveAlertGateFactory;
import com.atlassian.diagnostics.internal.platform.poller.DiagnosticPoller;

import javax.annotation.Nonnull;
import java.time.Clock;

public class DatabasePoolPoller extends DiagnosticPoller<DatabaseMonitorConfiguration> {

    private final DatabaseMonitor databaseMonitor;
    private final Clock clock;
    private final ConsecutiveAlertGate alertGate;
    private final DatabasePoolDiagnosticProvider databasePoolDiagnosticProvider;

    public DatabasePoolPoller(@Nonnull final DatabaseMonitorConfiguration config,
                              @Nonnull final DatabaseMonitor databaseMonitor,
                              @Nonnull final Clock clock,
                              @Nonnull final ConsecutiveAlertGateFactory alertGateFactory,
                              @Nonnull final DatabasePoolDiagnosticProvider databasePoolDiagnosticProvider) {
        super(DatabasePoolPoller.class.getName(), config);
        this.databaseMonitor = databaseMonitor;
        this.clock = clock;
        this.alertGate = alertGateFactory.createAlertGate(config::poolUtilizationTimeWindow, clock);
        this.databasePoolDiagnosticProvider = databasePoolDiagnosticProvider;
    }

    @Override
    protected void execute() {
        raiseAlertIfPoolUtilizationHasExceededThreshold(databasePoolDiagnosticProvider.getDiagnostic());
    }

    private void raiseAlertIfPoolUtilizationHasExceededThreshold(final DatabasePoolDiagnostic databasePoolDiagnostic) {
        if (alertGate.shouldRaiseAlert(() -> hasReachedDbPoolUtilisationThreshold(databasePoolDiagnostic)) && !databasePoolDiagnostic.isEmpty()) {
            databaseMonitor.raiseAlertForHighPoolUtilization(clock.instant(), databasePoolDiagnostic);
        }
    }

    private boolean hasReachedDbPoolUtilisationThreshold(final DatabasePoolDiagnostic databasePoolDiagnostic) {
        return (((double) databasePoolDiagnostic.getActiveConnections() / databasePoolDiagnostic.getMaxConnections()) * 100) >= monitorConfiguration.poolUtilizationPercentageLimit();
    }

}
