package com.atlassian.diagnostics.internal.platform.poller;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.stream.Collectors;

public class ScheduledPollerService {

    private static final Logger logger = LoggerFactory.getLogger(ScheduledPollerService.class);

    private final Map<ScheduledPoller, ScheduledFuture> pollers = new HashMap<>();
    private final ScheduledExecutorService scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();

    public synchronized void start(final List<ScheduledPoller> pollers) {
        for (final ScheduledPoller poller : pollers) {
            start(poller);
        }
    }

    public synchronized void start(final ScheduledPoller poller) {
        if (!pollers.containsKey(poller)) {
            schedulePoller(poller);
        } else {
            start(poller.diagnosticPoller.getKey());
        }
    }

    public synchronized void start(@Nonnull final String key) {
        findPoller(key).ifPresent(poller -> {
            final boolean canStart = Optional.ofNullable(pollers.get(poller))
                    .map(this::isCompleted)
                    .orElse(true);

            if (canStart) {
                schedulePoller(poller);
            }
        });
    }

    private void schedulePoller(final ScheduledPoller poller) {
        pollers.put(poller, scheduledExecutorService.scheduleAtFixedRate(
                run(poller),
                0,
                poller.scheduleInterval.getDelay(),
                poller.scheduleInterval.getTimeUnit()
        ));
    }

    private Runnable run(final ScheduledPoller poller) {
        return () -> {
            try {
                if (poller.diagnosticPoller.isEnabled()) {
                    poller.diagnosticPoller.execute();
                }
            } catch (Throwable e) {
                logger.debug("Poller failed", e);
            }
        };
    }

    private Optional<ScheduledPoller> findPoller(@Nonnull final String key) {
        return pollers.keySet().stream()
                .filter(poller -> poller.diagnosticPoller.getKey().equals(key))
                .findFirst();
    }

    public synchronized void shutdown() {
        scheduledExecutorService.shutdownNow();
    }

    public Set<PollerInfo> getPollerInfo() {
        return pollers.entrySet().stream()
                .map(poller -> new PollerInfo(
                        poller.getKey().diagnosticPoller.getKey(),
                        !isCompleted(poller.getValue())
                )).collect(Collectors.toSet());
    }

    private boolean isCompleted(final ScheduledFuture scheduledFuture) {
        return scheduledFuture == null || scheduledFuture.isDone() || scheduledFuture.isCancelled();
    }

    public static class ScheduledPoller {
        private final DiagnosticPoller diagnosticPoller;
        private final ScheduleInterval scheduleInterval;

        public static ScheduledPoller of(final DiagnosticPoller diagnosticPoller, final ScheduleInterval scheduleInterval) {
            return new ScheduledPoller(diagnosticPoller, scheduleInterval);
        }

        private ScheduledPoller(final DiagnosticPoller diagnosticPoller, final ScheduleInterval scheduleInterval) {
            this.diagnosticPoller = diagnosticPoller;
            this.scheduleInterval = scheduleInterval;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            ScheduledPoller that = (ScheduledPoller) o;
            return Objects.equals(diagnosticPoller.getKey(), that.diagnosticPoller.getKey());
        }

        @Override
        public int hashCode() {
            return Objects.hash(diagnosticPoller.getKey());
        }
    }
}
