package com.atlassian.diagnostics.internal.platform.monitor.event;

import com.atlassian.diagnostics.AlertTrigger;
import com.atlassian.diagnostics.util.CallingBundleResolver;
import com.atlassian.event.spi.ListenerHandler;
import com.atlassian.event.spi.ListenerInvoker;
import com.atlassian.plugin.osgi.util.OsgiHeaderUtil;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;

import java.util.Collections;
import java.util.List;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * {@link ListenerHandler handler} that adds monitoring support to the invokers created by the wrapped handler.
 */
class MonitoringListenerHandler implements ListenerHandler {

    private final CallingBundleResolver callingBundleResolver;
    private final ListenerHandler delegate;
    private final EventSystemMonitor monitor;
    private final String systemVersion;

    MonitoringListenerHandler(CallingBundleResolver callingBundleResolver, ListenerHandler delegate,
                              EventSystemMonitor monitor, String systemVersion) {
        this.callingBundleResolver = callingBundleResolver;
        this.delegate = delegate;
        this.monitor = monitor;
        this.systemVersion = systemVersion;
    }

    @Override
    public List<ListenerInvoker> getInvokers(Object listener) {
        if (listener == null) {
            return Collections.emptyList();
        }
        List<ListenerInvoker> invokers = delegate.getInvokers(listener);
        if (invokers == null) {
            return null;
        }

        AlertTrigger trigger = getTrigger(listener);
        return invokers.stream()
                .map(invoker -> new MonitoredListenerInvoker(monitor, trigger, invoker))
                .collect(collectingAndThen(toList(), Collections::unmodifiableList));
    }

    private AlertTrigger getTrigger(Object listener) {
        AlertTrigger.Builder builder = new AlertTrigger.Builder()
                .module(listener.getClass().getName());

        //determine the plugin that's registering the listener
        Bundle bundle = callingBundleResolver.getCallingBundle()
                .orElseGet(() -> FrameworkUtil.getBundle(listener.getClass()));
        if (bundle != null && bundle.getBundleId() != 0) {
            builder.plugin(OsgiHeaderUtil.getPluginKey(bundle), bundle.getVersion().toString());
        } else {
            builder.plugin("System", systemVersion);
        }
        return builder.build();
    }
}
