package com.atlassian.diagnostics.internal.platform.plugin;

import com.atlassian.annotations.VisibleForTesting;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;

public class PluginFinderImpl implements PluginFinder {
    private static final Logger logger = LoggerFactory.getLogger(PluginFinderImpl.class);

    private final ClassContextSecurityManager securityManger;
    private final BundleFinder bundleFinder;
    private final Cache<Class<?>, String> classPluginSourceCache;

    public PluginFinderImpl(final BundleFinder bundleFinder) {
        this(createClassContextSecurityManager(),
                bundleFinder,
                CacheBuilder.newBuilder()
                        .maximumSize(1000)
                        .expireAfterAccess(Duration.ofHours(1))
                        .build()
        );
    }

    private static ClassContextSecurityManager createClassContextSecurityManager() {
        try {
            return new ClassContextSecurityManager();
        } catch (Exception e) {
            logger.debug("Failed to create security manager", e);
            return null;
        }
    }

    @VisibleForTesting
    PluginFinderImpl(final ClassContextSecurityManager securityManger,
                     final BundleFinder bundleFinder,
                     final Cache<Class<?>, String> classPluginSourceCache) {
        this.securityManger = securityManger;
        this.bundleFinder = bundleFinder;
        this.classPluginSourceCache = classPluginSourceCache;
    }

    static class ClassContextSecurityManager extends SecurityManager {
        private static final Class<?>[] EMPTY_ARRAY = new Class<?>[0];

        @Override
        protected Class<?>[] getClassContext() {
            final Class<?>[] classContext = super.getClassContext();
            return classContext == null ? EMPTY_ARRAY : classContext;
        }
    }

    public Collection<String> getPluginNamesInCurrentCallStack() {
        try {
            final Set<String> plugins = new HashSet<>();
            if (securityManger != null) {
                for (final Class<?> clazz : securityManger.getClassContext()) {
                    final Optional<String> cachedPluginSource = Optional.ofNullable(classPluginSourceCache.getIfPresent(clazz));

                    if (!cachedPluginSource.isPresent()) {
                        resolvePlugin(clazz).ifPresent(pluginName -> add(plugins, pluginName));
                    } else if (isNotEmpty(cachedPluginSource.get())) {
                        add(plugins, cachedPluginSource.get());
                    }
                }
            }

            return plugins;
        } catch (Exception e) {
            logger.debug("Failed to get plugins list from call stack", e);
            return Collections.emptyList();
        }
    }

    private Optional<String> resolvePlugin(final Class<?> clazz) {
        final Optional<String> pluginName = bundleFinder.getBundleNameForClass(clazz);
        classPluginSourceCache.put(clazz, pluginName.orElse(""));

        return pluginName;
    }

    private void add(final Set<String> plugins, final String pluginName) {
        plugins.add(pluginName);
    }
}
