package com.atlassian.diagnostics.internal.platform.monitor.scheduler;

import com.atlassian.diagnostics.internal.platform.ConsecutiveAlertGate;
import com.atlassian.diagnostics.internal.platform.ConsecutiveAlertGateFactory;
import com.atlassian.diagnostics.internal.platform.poller.DiagnosticPoller;
import com.atlassian.scheduler.config.Schedule;
import com.atlassian.scheduler.status.JobDetails;
import com.atlassian.scheduler.status.RunDetails;

import javax.annotation.Nonnull;
import java.time.Clock;
import java.time.Instant;
import java.util.Collection;
import java.util.List;

public class SchedulerPoller extends DiagnosticPoller<SchedulerMonitorConfiguration> {

    private final SchedulerMonitor schedulerMonitor;
    private final SchedulerDiagnosticProvider diagnosticProvider;
    private final ConsecutiveAlertGate highUtilizationAlertGate;

    public SchedulerPoller(@Nonnull final SchedulerMonitorConfiguration config,
                           @Nonnull final SchedulerMonitor schedulerMonitor,
                           @Nonnull final SchedulerDiagnosticProvider diagnosticProvider,
                           @Nonnull final Clock clock,
                           @Nonnull final ConsecutiveAlertGateFactory alertGateFactory) {
        super(SchedulerPoller.class.getName(), config);
        this.schedulerMonitor = schedulerMonitor;
        this.diagnosticProvider = diagnosticProvider;
        this.highUtilizationAlertGate = alertGateFactory.createAlertGate(config::highUtilizationTimeWindow, clock);
    }

    @Override
    protected void execute() {
        final SchedulerDiagnostic diagnostic = diagnosticProvider.getDiagnostic();
        raiseAlertIfWorkerThreadsHighlyUtilized(diagnostic.getRunningJobs(), diagnostic.getWorkerThreads());
        raiseAlertIfJobExecutionExceededInterval(diagnostic.getScheduledJobs());
    }

    private void raiseAlertIfWorkerThreadsHighlyUtilized(final List<RunningJobDiagnostic> runningJobs, final int workerThreadCount) {
        if (highUtilizationAlertGate.shouldRaiseAlert(() -> runningJobs.size() >= workerThreadCount)) {
            schedulerMonitor.raiseAlertForHighUtilization(Instant.now(), runningJobs, workerThreadCount);
        }
    }

    private void raiseAlertIfJobExecutionExceededInterval(final Collection<ScheduledJobDiagnostic> scheduledJobs) {
        scheduledJobs.forEach(scheduledJob ->
            scheduledJob.getLastRun().ifPresent(lastRun -> {
                if (lastRunDurationExceededInterval(scheduledJob.getJobDetails(), lastRun)) {
                    schedulerMonitor.raiseAlertForSlowJob(Instant.now(), scheduledJob);
                }
            })
        );
    }

    private boolean lastRunDurationExceededInterval(final JobDetails jobDetails, final RunDetails lastRun) {
        final Schedule schedule = jobDetails.getSchedule();
        return isIntervalSchedule(schedule) &&
                isLastRunDurationLongerThanScheduledInterval(schedule, lastRun);
    }

    private boolean isIntervalSchedule(final Schedule schedule) {
        return Schedule.Type.INTERVAL == schedule.getType();
    }

    private boolean isLastRunDurationLongerThanScheduledInterval(final Schedule schedule, final RunDetails lastRun) {
        return schedule.getIntervalScheduleInfo().getIntervalInMillis() > 0 &&
                lastRun.getDurationInMillis() >= schedule.getIntervalScheduleInfo().getIntervalInMillis();
    }

}
