package com.atlassian.diagnostics.internal.platform.monitor.db.pool.resolver;

import com.atlassian.diagnostics.internal.platform.monitor.db.pool.DatabasePoolDiagnostic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.management.Attribute;
import javax.management.ObjectInstance;
import java.lang.management.ManagementFactory;
import java.time.Duration;
import java.util.List;
import java.util.Optional;

public class JmxDatabasePoolBean {
    private static final Logger logger = LoggerFactory.getLogger(JmxDatabasePoolBean.class);

    private final JmxDatabasePoolAttributes jmxAttributes;
    private final ObjectInstance objectInstance;

    JmxDatabasePoolBean(final JmxDatabasePoolAttributes jmxAttributes, final ObjectInstance objectInstance) {
        this.jmxAttributes = jmxAttributes;
        this.objectInstance = objectInstance;
    }

    public Optional<DatabasePoolDiagnostic> generateDatabasePoolMetrics() {
        logger.trace("Class Name: [{}]", objectInstance.getClassName());
        final List<Attribute> resolvedBeanAttributes = jmxAttributes.getJmxAttributes(ManagementFactory.getPlatformMBeanServer(), objectInstance);
        try {
            if (!resolvedBeanAttributes.isEmpty()) {
                logger.trace("Found DB Pool values: [{}]", resolvedBeanAttributes);
                final Integer idleConnections = getMBeanAttributeValue(resolvedBeanAttributes, JmxDatabasePoolAttributes.IDLE_CONNECTION_ATTRIBUTE_INDEX);
                final Integer activeConnections = getMBeanAttributeValue(resolvedBeanAttributes, JmxDatabasePoolAttributes.ACTIVE_CONNECTION_ATTRIBUTE_INDEX);
                final Integer maxConnections = getMBeanAttributeValue(resolvedBeanAttributes, JmxDatabasePoolAttributes.MAX_CONNECTION_ATTRIBUTE_INDEX);
                return Optional.of(new DatabasePoolDiagnostic(idleConnections, activeConnections, maxConnections));
            }
        } catch (final Exception exception) {
            logger.error("Error querying for JMX attributes: [{}] on MBean: [{}]", jmxAttributes.poolUseAttributeNames, objectInstance, exception);
        }

        return Optional.empty();
    }

    private Integer getMBeanAttributeValue(final List<Attribute> resolvedBeanAttributes, final int index) {
        final Integer value = (Integer) resolvedBeanAttributes.get(index).getValue();
        if (value == null) {
            throw new NullPointerException("beanAttribute value was null");
        } else {
            return value;
        }
    }

    public Duration getPoolConnectionLeakTimeout() {
        try {
            final Object abandonedTimeoutAttribute = ManagementFactory.getPlatformMBeanServer()
                    .getAttribute(objectInstance.getObjectName(), jmxAttributes.abandonedTimeoutAttributeName);

            if (abandonedTimeoutAttribute instanceof Integer) {
                final Integer abandonedTimeoutValue = (Integer) abandonedTimeoutAttribute;
                if (abandonedTimeoutValue != Integer.MAX_VALUE) {
                    return Duration.ofSeconds(abandonedTimeoutValue);
                }
            }
        } catch (final Exception e) {
            logger.debug("Returning default DbPoolMetrics", e);
        }

        return Duration.ofMillis(0);
    }
}
