package com.atlassian.diagnostics.internal.platform.monitor.db.pool.resolver;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.management.Attribute;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanServer;
import javax.management.ObjectInstance;
import javax.management.ReflectionException;
import java.util.Collections;
import java.util.List;

public enum JmxDatabasePoolAttributes {
    APACHE_COMMONS(
            "org.apache.commons.pool2.impl.GenericObjectPool",
            PoolAttributeBuilder.builder()
                    .idleConnectionsAttribute("NumIdle")
                    .activeConnectionsAttribute("NumActive")
                    .maxConnectionsAttribute("MaxTotal")
                    .build(),
            "RemoveAbandonedTimeout"
    ),
    TOMCAT(
            "org.apache.tomcat.dbcp.pool2.impl.GenericObjectPool",
            PoolAttributeBuilder.builder()
                    .idleConnectionsAttribute("NumIdle")
                    .activeConnectionsAttribute("NumActive")
                    .maxConnectionsAttribute("MaxTotal")
                    .build(),
            "RemoveAbandonedTimeout"
    ),
    VIBUR(
            "org.vibur.dbcp.ViburDBCPMonitoring",
            PoolAttributeBuilder.builder()
                    .idleConnectionsAttribute("PoolRemainingCreated")
                    .activeConnectionsAttribute("PoolTaken")
                    .maxConnectionsAttribute("PoolMaxSize")
                    .build()
    ),
    HIKARI(
            "com.zaxxer.hikari.pool.HikariPool",
            PoolAttributeBuilder.builder()
                    .idleConnectionsAttribute("IdleConnections")
                    .activeConnectionsAttribute("ActiveConnections")
                    .maxConnectionsAttribute("TotalConnections")
                    .build()
    );

    public static final int IDLE_CONNECTION_ATTRIBUTE_INDEX = 0;
    public static final int ACTIVE_CONNECTION_ATTRIBUTE_INDEX = 1;
    public static final int MAX_CONNECTION_ATTRIBUTE_INDEX = 2;

    private static final Logger logger = LoggerFactory.getLogger(JmxDatabasePoolAttributes.class);

    public final String instanceOfQuery;
    public final String[] poolUseAttributeNames;
    public final String abandonedTimeoutAttributeName;

    JmxDatabasePoolAttributes(final String instanceOfQuery, final String[] poolUseAttributeNames) {
        this(instanceOfQuery, poolUseAttributeNames, "");
    }

    JmxDatabasePoolAttributes(final String instanceOfQuery, final String[] poolUseAttributeNames, final String abandonedTimeoutAttributeName) {
        this.instanceOfQuery = instanceOfQuery;
        this.poolUseAttributeNames = poolUseAttributeNames;
        this.abandonedTimeoutAttributeName = abandonedTimeoutAttributeName;
    }

    public List<Attribute> getJmxAttributes(@Nonnull final MBeanServer beanServer, @Nonnull final ObjectInstance objectInstance) {
        try {
            return beanServer.getAttributes(objectInstance.getObjectName(), this.poolUseAttributeNames).asList();
        } catch (InstanceNotFoundException | ReflectionException e) {
            logger.debug("Failed to get jmxAttributes for {}", objectInstance);
            return Collections.emptyList();
        }
    }

    private static class PoolAttributeBuilder {
        private String idleConnections;
        private String activeConnections;
        private String maxConnections;

        private static PoolAttributeBuilder builder() {
            return new PoolAttributeBuilder();
        }

        private PoolAttributeBuilder() {}

        private PoolAttributeBuilder idleConnectionsAttribute(final String idleConnections) {
            this.idleConnections = idleConnections;
            return this;
        }

        private PoolAttributeBuilder activeConnectionsAttribute(final String activeConnections) {
            this.activeConnections = activeConnections;
            return this;
        }

        private PoolAttributeBuilder maxConnectionsAttribute(final String maxConnections) {
            this.maxConnections = maxConnections;
            return this;
        }

        public String[] build() {
            return new String[] { idleConnections, activeConnections, maxConnections };
        }
    }
}
