package com.atlassian.diagnostics.ipd.api.registry;

import com.atlassian.diagnostics.ipd.api.MeterConfigurations;
import com.atlassian.diagnostics.ipd.api.jobs.IpdJobRegistry;
import org.osgi.framework.BundleContext;

/**
 * This interface should be imported by a plugin that wants to expose metrics via IPD library.
 * This interface creates plugin-specific instance of IpdRegistry. Plugin meters are stored in global product IpdRegistry,
 * enforcing meters to be unique across products and plugins.
 * <p>
 * <b>Important Note - meters lifecycle:</b><br/>
 * To avoid meters inconsistency due to plugin reloads and upgrades, the plugin should always cleanup its meters.
 * For safe cleanup of meters, the plugin has to either: <br/>
 * - Register created instance of IpdRegistry as a Bean in Spring context and let Spring manage its lifecycle.<br/>
 * - Manually call {@link IpdRegistry#shutdown()} method when the plugin is being shutdown.
 * </p>
 * @since 5.0.0
 */
public interface PluginIpdFactory {
    /**
     * Creates a new instance of IpdRegistry specific for the plugin.
     * <p>
     *     All meters created with this instance will apply configurations defined in the product.
     * </p>
     * <p>
     *  <b>Important Note - meters lifecycle:</b><br/>
     *  To avoid meters inconsistency due to plugin reloads and upgrades, the plugin should always cleanup its meters.
     *  For safe cleanup of meters, the plugin has to either: <br/>
     *  - Register created instance of IpdRegistry as a Bean in Spring context and let Spring manage its lifecycle.<br/>
     *  - Manually call {@link IpdRegistry#shutdown()} method when the plugin is being shutdown.
     *  </p>
     * @param bundleContext the bundle context of the plugin
     * @return a new instance of IpdRegistry
     */
    IpdRegistry createRegistry(BundleContext bundleContext);

    /**
     * Creates a new instance of IpdRegistry specific for the plugin.
     * <p>
     *     All meters created with this instance will:<br/>
     *     1. apply configurations defined in the product <br/>
     *     2. override product configurations with plugin-specific configurations provided in the `meterConfig` parameter.
     * </p>
     * <p>
     *  <b>Important Note - meters lifecycle:</b><br/>
     *  To avoid meters inconsistency due to plugin reloads and upgrades, the plugin should always cleanup its meters.
     *  For safe cleanup of meters, the plugin has to either: <br/>
     *  - Register created instance of IpdRegistry as a Bean in Spring context and let Spring manage its lifecycle.<br/>
     *  - Manually call {@link IpdRegistry#shutdown()} method when the plugin is being shutdown.
     *  </p>
     * @param bundleContext the bundle context of the plugin
     * @return a new instance of IpdRegistry
     */
    IpdRegistry createRegistry(MeterConfigurations meterConfig, BundleContext bundleContext);

    /**
     * Creates a new instance of IpdJobRegistry specific for the plugin.
     * <p>
     *     All jobs registered with this instance will run in regular interval until the plugin is disabled/uninstalled.
     * </p>
     * <p>
     *  <b>Important Note - jobs lifecycle:</b><br/>
     *  To avoid inconsistency due to plugin reloads and upgrades, the plugin should always cleanup its jobs.
     *  For safe cleanup of jobs, the plugin has to either: <br/>
     *  - Register created instance of IpdJobsRegistry as a Bean in Spring context and let Spring manage its lifecycle.<br/>
     *  - Manually call {@link IpdJobRegistry#shutdown()} method when the plugin is being shutdown.
     *  </p>
     * @param bundleContext the bundle context of the plugin
     * @return a new instance of IpdJobRegistry
     */
    IpdJobRegistry createJobRegistry(BundleContext bundleContext);
}
