package com.atlassian.diagnostics.ipd.api.meters.custom.type;

import com.atlassian.diagnostics.ipd.api.meters.CustomMeter;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * Metric type holding information about the number of values that fall into specific buckets.
 * <p>
 *     Example:
 * </p>
 * <pre>
 * the buckets are defined as "a", "b", "c"
 * increment("a") -> bucket "a" will be incremented
 * increment("b") -> bucket "b" will be incremented
 * increment("invalid") -> nothing will be incremented
 * </pre>
 *
 * Made for use with {@link CustomMeter}.
 * @since 5.0.0
 */
public class IpdStringBucketsCounter implements IpdBucketsCounterMxBean {
    private final Map<String, AtomicLong> bucketsMap;
    public IpdStringBucketsCounter(final String... buckets) {
        this.bucketsMap = Arrays.stream(buckets)
                .collect(Collectors.toUnmodifiableMap(b -> b, b -> new AtomicLong()));
    }

    public void increment(@Nonnull String value) {
        final var counter = bucketsMap.get(value);
        if (counter != null) {
            counter.incrementAndGet();
        }
    }

    public void reset() {
        for (AtomicLong bucket : bucketsMap.values()) {
            bucket.set(0);
        }
    }

    @Override
    public Map<String, Long> getBuckets() {
        return Collections.unmodifiableMap(
                bucketsMap.entrySet()
                        .stream()
                        .collect(Collectors.toMap(
                                Map.Entry::getKey,
                                e -> e.getValue().get())));
    }

    public static class StringBucketsDefinition extends CustomMeter.MBeanSupplier<IpdStringBucketsCounter> {
        private StringBucketsDefinition(String... buckets) {
            super(IpdStringBucketsCounter.class, () -> new IpdStringBucketsCounter(buckets));
        }
    }

    public static StringBucketsDefinition buckets(String... buckets) {
        return new StringBucketsDefinition(buckets);
    }
}
