package com.atlassian.diagnostics.ipd.api.meters.custom.type;

import com.atlassian.diagnostics.ipd.api.meters.CustomMeter;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Metric type holding information about the number of values that fall into specific buckets.
 * <p>
 *     Example:
 * </p>
 * <pre>
 * the buckets are defined as 1, 10, 100
 * putValue(5) -> bucket 10 will be incremented
 * putValue(10) -> bucket 100 will be incremented
 * putValue(101) -> bucket "MAX" will be incremented
 * </pre>
 *
 * Made for use with {@link CustomMeter}.
 * @since 5.0.0
 */
public class IpdLongBucketsCounter implements IpdBucketsCounterMxBean {
    private static final Long[] MAX_LONG = {Long.MAX_VALUE};

    private final Map<Long, AtomicLong> bucketsMap;
    private final Long[] buckets;
    public IpdLongBucketsCounter(final Long... buckets) {
        this.buckets = Arrays.stream(buckets).sorted().toArray(Long[]::new);
        this.bucketsMap = Stream.of(this.buckets, MAX_LONG)
                .flatMap(Arrays::stream)
                .collect(Collectors.toUnmodifiableMap(b -> b, b -> new AtomicLong()));
    }

    public void putValue(final long value) {
        for (long bucket : buckets) {
            if (value < bucket) {
                bucketsMap.get(bucket).incrementAndGet();
                return;
            }
        }
        bucketsMap.get(Long.MAX_VALUE).incrementAndGet();
    }

    public void reset() {
        for (AtomicLong bucket : bucketsMap.values()) {
            bucket.set(0);
        }
    }

    @Override
    public Map<String, Long> getBuckets() {
        return Collections.unmodifiableMap(
                bucketsMap.entrySet()
                        .stream()
                        .collect(Collectors.toMap(
                                entry -> {
                                    if (entry.getKey() == Long.MAX_VALUE) {
                                        return "MAX";
                                    }
                                    return entry.getKey().toString();
                                },
                                e -> e.getValue().get())));
    }

    public static class LongBucketsDefinition extends CustomMeter.MBeanSupplier<IpdLongBucketsCounter> {
        private LongBucketsDefinition(Long... buckets) {
            super(IpdLongBucketsCounter.class, () -> new IpdLongBucketsCounter(buckets));
        }
    }

    public static LongBucketsDefinition buckets(Long... buckets) {
        return new LongBucketsDefinition(buckets);
    }
}
