package com.atlassian.diagnostics.ipd.api.meters.custom;


import com.atlassian.diagnostics.ipd.api.meters.AbstractIpdMeter;
import com.atlassian.diagnostics.ipd.api.meters.CustomMeter;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterFactory;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import org.slf4j.Logger;

import javax.management.MBeanServer;
import javax.management.MXBean;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

import static java.lang.management.ManagementFactory.getPlatformMBeanServer;
import static org.slf4j.LoggerFactory.getLogger;

public class CustomMeterImpl<T> extends AbstractIpdMeter implements CustomMeter<T> {

    private static final String TYPE_ID = "custom";

    private static final Logger LOG = getLogger(CustomMeterImpl.class);
    private static final String SINGLE_MX_BEAN_REQUIRED_EXCEPTION = "IpdCustomMetric requires the Type %s to implement exactly one interface with @MXBean annotation.";
    private static final TypeReference<Map<String, Object>> MAP_TYPE_REFERENCE = new TypeReference<Map<String, Object>>(){};
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private final Class<?> beanType;
    private final MBeanServer mBeanServer;
    private final AtomicBoolean jmxRegistered = new AtomicBoolean(false);
    private final ObjectWriter objectWriter;
    private final T mBean;
    private final String typeId;

    protected CustomMeterImpl(final MeterConfig config, final T mBean) {
        this(config, mBean, getPlatformMBeanServer());
    }

    protected CustomMeterImpl(final MeterConfig config,
                              final T mBean,
                              final MBeanServer mBeanServer) {
        super(config);
        this.beanType = mBean.getClass();
        this.mBean = mBean;
        this.mBeanServer = mBeanServer;
        final Class<?> interfaceType = findMBeanInterface(beanType);
        this.objectWriter = OBJECT_MAPPER.writerFor(interfaceType);
        this.typeId = getTypeId(beanType);
        if (isEnabled()) {
            registerMBean();
        }
    }

    private Class<?> findMBeanInterface(Class<?> type) {
        return Arrays.stream(type.getInterfaces())
                .filter(i -> i.isAnnotationPresent(MXBean.class))
                .reduce((a, b) -> {
                    // If there is more than one MXBean interface
                    throw new IpdCustomMetricRegisterException(String.format(SINGLE_MX_BEAN_REQUIRED_EXCEPTION, type.getName()));
                })
                .orElseThrow(() -> new IpdCustomMetricRegisterException(String.format(SINGLE_MX_BEAN_REQUIRED_EXCEPTION, type.getName())));
    }

    @Override
    public T getMBeanObject() {
        return mBean;
    }

    @Override
    public Map<String, Object> getAttributes(boolean extraAttributes) {
        // This makes sure we serialize only properties from the MBean interface
        try {
            final String attributes = objectWriter.writeValueAsString(mBean);
            return OBJECT_MAPPER.readValue(attributes, MAP_TYPE_REFERENCE);
        } catch (IOException e) {
            throw new IpdCustomMetricRegisterException("Couldn't read attributes from MBean for IpdCustomMetric with type " + beanType.getName(), e);
        }
    }

    @Override
    public boolean isVisible() {
        return isEnabled() && jmxRegistered.get();
    }

    @Override
    protected void registerMBean() {
        if (!isEnabled()) {
            return;
        }
        if (!jmxRegistered.compareAndSet(false, true)) {
            return;
        }
        try {
            mBeanServer.registerMBean(mBean, getObjectName());
        } catch (Exception e) {
            throw new IpdCustomMetricRegisterException("Exception occurred while registering MBean for IpdCustomMetric with type " + beanType.getName(), e);
        }
    }

    @Override
    protected void unregisterMBean() {
        try {
            if (jmxRegistered.compareAndSet(true, false)) {
                mBeanServer.unregisterMBean(getObjectName());
            }
        } catch (Exception e) {
            throw new IpdCustomMetricRegisterException(String.format("Failed to unregister metric %s of type %s", getMeterKey(), beanType));
        }
    }

    @Override
    public String getTypeId() {
        return typeId;
    }

    @Override
    public void update(final Consumer<T> updater) {
        if (!isEnabled()) {
            return;
        }
        registerMBean();

        updater.accept(mBean);
        metricUpdated();
    }

    private static String getTypeId(final Class<?> beanType) {
        return TYPE_ID + "#" + beanType.getName();
    }

    public static <BEAN_TYPE> MeterFactory<CustomMeter<BEAN_TYPE>> factory(final Class<BEAN_TYPE> beanType) {
        return factory(new MBeanSupplier<>(beanType));
    }

    public static <BEAN_TYPE> MeterFactory<CustomMeter<BEAN_TYPE>> factory(final MBeanSupplier<BEAN_TYPE> mBeanSupplier) {
        return new MeterFactory<>(
                config -> new CustomMeterImpl<>(config, mBeanSupplier.createInstance()),
                getTypeId(mBeanSupplier.getBeanType()),
                "custom");
    }
}
