package com.atlassian.diagnostics.ipd.api.meters.config;

import com.atlassian.diagnostics.ipd.api.MeterKey;
import com.atlassian.diagnostics.ipd.api.meters.IpdMeter;
import org.slf4j.Logger;

import javax.management.ObjectName;
import java.util.Collections;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * Immutable configuration for an IPD meter.
 * <p>
 *     Meters configuration should be configured through the
 *     {@link com.atlassian.diagnostics.ipd.api.MeterConfigurations#addMeterConfig(String, Consumer)} method.
 * </p>
 * @since 5.0.0
 */
public class MeterConfig {
    private final MeterKey meterKey;
    private final ObjectName objectName;
    private final Map<String, Object> properties;
    private final boolean workInProgress;
    private final boolean logOnUpdate;
    private final boolean intervalLogging;
    private final Logger logger;
    private final Supplier<Boolean> enabledCheck;
    private final Predicate<IpdMeter> loggingCondition;
    private final Consumer<IpdMeter> updateListener;

    public MeterConfig(final MeterKey meterKey, final MeterConfigBuilderImpl meterConfigBuilder) {
        this.meterKey = meterKey;
        this.objectName = meterConfigBuilder.objectNameFactory.constructObjectName(meterKey);
        this.enabledCheck = () -> meterConfigBuilder.enabledCheck.test(this);
        this.properties = Collections.unmodifiableMap(meterConfigBuilder.properties);
        this.workInProgress = this.properties.getOrDefault(Properties.WORK_IN_PROGRESS, false).equals(true);
        this.logOnUpdate = this.properties.getOrDefault(Properties.LOG_ON_UPDATE, false).equals(true);
        this.intervalLogging = this.properties.getOrDefault(Properties.INTERVAL_LOGGING, true).equals(true);
        this.logger = meterConfigBuilder.logger;
        this.loggingCondition = meterConfigBuilder.shouldLog;
        this.updateListener = logOnUpdate ? meterConfigBuilder.updateListener : (ignored) -> {};
    }

    /**
     * Returns a supplier that checks if the meter is enabled.
     * @return Supplier that checks if the meter is enabled
     */
    public Supplier<Boolean> getEnabledCheck() {
        return enabledCheck;
    }

    /**
     * Returns true if the meter is marked as "work in progress" Work in progress meters require additional feature flag to be enabled.
     * @return true if the meter is marked as "work in progress"
     */
    public boolean isWorkInProgress() {
        return workInProgress;
    }

    /**
     * Returns a listener that is called when the meter value is updated.
     * @return Consumer that is called when the meter value is updated
     */
    public Consumer<IpdMeter> getMetricUpdateListener() {
        return updateListener;
    }

    /**
     * Returns the meter key.
     * @return Meter key
     */
    public MeterKey getMeterKey() {
        return meterKey;
    }

    /**
     * Returns the object name of the MBean. The object name is used to query the metric value in JMX.
     * @return Object name of the MBean
     */
    public ObjectName getObjectName() {
        return objectName;
    }

    /**
     * Returns the unmodifiable properties map of the meter.
     * The properties map is used to store additional configuration of the meter.
     * @return Properties map of the meter
     */
    public Map<String, Object> getProperties() {
        return properties;
    }

    /**
     * Returns true if the meter should log on value update.
     * @return true if the meter should log on value update
     */
    public boolean isLogOnUpdate() {
        return logOnUpdate;
    }

    /**
     * Returns true if the meter should log at regular intervals.
     * @return true if the meter should log at regular intervals
     */
    public boolean isIntervalLoggingEnabled() {
        return intervalLogging;
    }

    /**
     * Returns the logging condition of the meter.
     * The logging condition is used to determine if the meter should log at a given time.
     * @return Logging condition of the meter
     */
    public Predicate<IpdMeter> getLoggingCondition() {
        return loggingCondition;
    }

    /**
     * Returns the logger for this meter, logger defines to which file meter value will be logged.
     * The logger is used to log messages from the meter.
     * @return Logger of the meter
     */
    public Logger getLogger() {
        return logger;
    }

    /**
     * Common property keys used in the properties map of the meter.
     */
    public static class Properties {

        public static final String WORK_IN_PROGRESS = "workInProgress";
        public static final String LOG_ON_UPDATE = "logOnUpdate";
        public static final String INTERVAL_LOGGING = "intervalLogging";
        public static final String CUSTOM_CLOCK = "customClock"; // Used for testing purposes

        private Properties() {
        }
    }
}
