package com.atlassian.diagnostics.ipd.api.meters;

import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterFactory;

import java.util.concurrent.TimeUnit;

/**
 * Interface representing a statistics meter, which is a type of {@link IpdMeter}.
 * A statistics meter can be updated with a single long value, which is used to calculate historical statistics.
 * <p>
 *     This meter will calculate mean, min, max, p50, p90, p99 and more.
 *     All statistics are calculated only for recent values (e.g. from last 5-10 minutes)
 *     and can't be relied on for calculating statistics of all meters updates.
 * </p>
 * @since 3.0.0
 */
public interface StatsMeter extends IpdMeter {
    String TYPE_ID = "stats";
    MeterFactory<StatsMeter> NOOP_FACTORY = new MeterFactory<>(Noop::new, TYPE_ID, "statistics");

    @Override
    default String getTypeId() {
        return TYPE_ID;
    }

    /**
     * Updates the metric with the given value. The time unit of the value is milliseconds.
     *
     * @param value value for updated statistics
     */
    default void update(final long value) {
        update(value, TimeUnit.MILLISECONDS);
    }

    /**
     * Updates the metric with the given value. You may specify the time unit of the value.
     *
     * @param value    value for updated statistics
     * @param timeUnit time unit of the value
     */
    void update(final long value, final TimeUnit timeUnit);

    class Noop extends NoopMeter implements StatsMeter {

        protected Noop(final MeterConfig meterConfig) {
            super(meterConfig);
        }

        @Override
        public void update(final long value, final TimeUnit timeUnit) {
            metricUpdated();
        }
    }
}
