package com.atlassian.diagnostics.ipd.api.meters;

import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfig;
import com.atlassian.diagnostics.ipd.api.meters.config.MeterFactory;
import com.atlassian.diagnostics.ipd.api.meters.custom.IpdCustomMetricRegisterException;
import com.atlassian.diagnostics.ipd.api.meters.custom.type.IpdLongBucketsCounter;
import com.atlassian.diagnostics.ipd.api.meters.custom.type.IpdStringBucketsCounter;

import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * Interface representing a custom meter, which is a type of {@link IpdMeter}.
 * A custom meter is used to expose custom attributes to JMX and log file through the IPD framework.
 * <p>
 *     You can define custom attributes by creating a class which implements an interface annotated with {@link javax.management.MXBean}.
 *     Getters on this interface define the exposed JMX attributes. <br>
 *     <b>Your class implementation must be thread safe.</b>
 * </p>
 * See example implementations: <br/>
 * {@link com.atlassian.diagnostics.ipd.api.meters.custom.type.IpdConnectionState}<br/>
 * {@link IpdLongBucketsCounter}
 *
 * @param <T> T is a class which implements an interface annotated with {@link javax.management.MXBean}. Getters on this interface define the exposed JMX attributes.
 */
public interface CustomMeter<T> extends IpdMeter {
    String TYPE_ID = "custom";
    static <T> MeterFactory<CustomMeter<T>> noopFactory(final MBeanSupplier<T> mBeanSupplier) {
        return new MeterFactory<>(config -> new Noop<>(mBeanSupplier.createInstance(), config), TYPE_ID, "custom");
    }

    /**
     *
     * The returned object holds attribute data that are exposed to JMX.
     * You can update the meter state directly by calling methods on this instance.
     * <p>However, any changes made through this instance will bypass enabled checks of this meter. <br/>
     * <b>Direct updates won't be logged to file when the meter is configured as
     * {@link com.atlassian.diagnostics.ipd.api.meters.config.MeterConfigBuilder#setLogOnUpdate(boolean)}.</b>
     * </p>
     *
     * <b>Do not keep this MBeanObject instance long-term.</b>
     * @return the object that will be exposed to JMX
     */
    T getMBeanObject();

    /**
     * Updates the meter state by calling the updater function.
     * Updater function will be called only when the meter is enabled.
     * <p>Updates made through this instance will be logged to file when the meter is configured as
     * {@link com.atlassian.diagnostics.ipd.api.meters.config.MeterConfigBuilder#setLogOnUpdate(boolean)}.</p>
     * @param updater function that updates the meter state
     */
    void update(Consumer<T> updater);

    /**
     * Helper class which supplies a new instance of CustomMBean.
     * See example implementation: <br/>
     * {@link IpdLongBucketsCounter.LongBucketsDefinition}
     * {@link IpdStringBucketsCounter.StringBucketsDefinition}
     * @param <BEAN_TYPE>
     */
    class MBeanSupplier<BEAN_TYPE> {
        private final Class<BEAN_TYPE> beanType;
        private final Supplier<BEAN_TYPE> mBeanSupplier;

        /**
         * A supplier which creates a new instance of CustomMBean with a custom supplier.
         * @param beanType class of the CustomMBean
         * @param mBeanSupplier supplier of the CustomMBean
         */
        public MBeanSupplier(Class<BEAN_TYPE> beanType, Supplier<BEAN_TYPE> mBeanSupplier) {
            this.beanType = beanType;
            this.mBeanSupplier = mBeanSupplier;
        }

        /**
         * A supplier which creates a new instance of CustomMBean with default constructor.
         * @param beanType class of the CustomMBean, <b>must have a default constructor</b>
         */
        public MBeanSupplier(Class<BEAN_TYPE> beanType) {
            this(beanType, () -> {
                try {
                    return beanType.getDeclaredConstructor().newInstance();
                } catch (Exception e) {
                    throw new IpdCustomMetricRegisterException(String.format("Couldn't create instance with default constructor for metric of type %s", beanType.getName()), e);
                }
            });
        }

        public Class<BEAN_TYPE> getBeanType() {
            return beanType;
        }

        public BEAN_TYPE createInstance() {
            return mBeanSupplier.get();
        }
    }

    class Noop<T> extends NoopMeter implements CustomMeter<T> {

        private final T instance;

        public Noop(final T instance, final MeterConfig meterConfig) {
            super(meterConfig);
            this.instance = instance;
        }
        @Override
        public T getMBeanObject() {
            return instance;
        }

        @Override
        public String getTypeId() {
            return TYPE_ID;
        }

        @Override
        public void update(Consumer<T> updater) {
            updater.accept(instance);
            metricUpdated();
        }
    }
}
