package com.atlassian.diagnostics.ipd.api.jobs;

import javax.annotation.Nonnull;

/**
 * IPD job registry that will schedule {@link IpdJob}s.
 * Registered jobs are triggered on a regular interval.
 *
 * @since 5.0.0
 */
public interface IpdJobRegistry {

    /**
     * Register single job. Only a single job can be registered for a single job class name.
     * <p>
     *     Overrides existing job if it was registered before.
     * </p>
     * @param job job to register
     * @param key unique key for the job, should be dot-separated string e.g. <b>database.connection.latency</b>
     */
    void register(@Nonnull IpdJob job, @Nonnull String key);

    /**
     * Unregisters a single job.
     */
    default void unregister(@Nonnull IpdJob job) {
        unregister(job.getClass());
    }

    /**
     * Unregisters a single job by its class.
     */
    void unregister(@Nonnull Class<? extends IpdJob> jobClass);

    /**
     * Register single job with options. Only a single job can be registered for a single job class name.
     * <p>
     *      Overrides existing job if it was registered before.
     * </p>
     */
    void register(@Nonnull IpdJobRegistration jobRegistration);

    /**
     * Unregisters all jobs.
     */
    void shutdown();
}
