package com.atlassian.diagnostics.ipd.api;

import java.util.Objects;

/**
 * Represents a tag for a meter.
 * A tag consists of a key-value pair, where both the key and value are strings.
 * This class provides methods to retrieve the key and value, as well as to format the tag.
 * Use the {@link #of(String, String)} method to create a new MeterTag.
 * @since 5.0.0
 */
public class MeterTag {
    private final String key;
    private final String value;

    private MeterTag(String key, String value) {
        this.key = key;
        this.value = value;
    }

    public String getKey() {
        return this.key;
    }

    public String getValue() {
        return this.value;
    }

    /**
     * Returns a new MeterTag with the key and value formatted to replace spaces with underscores.
     * This is useful for displaying the tag in JMX and log files.
     * @return a new MeterTag with the key and value formatted to replace spaces with underscores
     */
    public MeterTag getFormattedTag() {
        return MeterTag.of(
                key.replace(" ", "_"),
                value.replace(" ", "_"));
    }

    /**
     * Creates a new MeterTag with the given key and value.
     * @param key the key of the tag
     * @param value the value of the tag
     * @return a new MeterTag with the given key and value
     */
    public static MeterTag of(String key, String value) {
        return new MeterTag(key, value);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        MeterTag meterTag = (MeterTag) o;
        return Objects.equals(key, meterTag.key) && Objects.equals(value, meterTag.value);
    }

    @Override
    public int hashCode() {
        return Objects.hash(key, value);
    }

    @Override
    public String toString() {
        return key + "=" + value;
    }
}
