package com.atlassian.diagnostics.ipd.api;

import com.google.common.collect.ImmutableMap;

import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;

import static java.util.stream.Collectors.toMap;

/**
 * Represents a key for an IPD meter, which includes a name and an array of tags. Tags are meters dimensions.
 * This class provides methods to retrieve the name, tags, and their corresponding formats.
 * @since 5.0.0
 */
public class MeterKey {

    private final String name;
    private final MeterTag[] tags;
    private final Map<String, String> tagsMap;
    private final Map<String, String> formattedTagsMap;

    public MeterKey(final String name, final MeterTag... tags) {
        this.name = name;
        this.tags = tags;
        this.tagsMap = ImmutableMap.copyOf(Arrays.stream(tags)
                .collect(toMap(MeterTag::getKey, MeterTag::getValue)));
        this.formattedTagsMap = ImmutableMap.copyOf(Arrays.stream(tags)
                .map(MeterTag::getFormattedTag)
                .collect(toMap(MeterTag::getKey, MeterTag::getValue)));
    }

    public String getName() {
        return name;
    }

    public MeterTag[] getTags() {
        return tags;
    }

    public Map<String, String> getTagsAsMap() {
        return tagsMap;
    }

    /**
     * The formatted tag values are displayed in JMX and log files. See {@link MeterTag#getFormattedTag()}
     * @return the tags as a map with the tag keys as the keys and the formatted tag values as the values
     */
    public Map<String, String> getFormattedTagsAsMap() {
        return formattedTagsMap;
    }

    // This should be quicker than Map if there are none or only a few tags
    @Nullable
    public String getTag(final String key) {
        for (final MeterTag tag : tags) {
            if (tag.getKey().equals(key)) {
                return tag.getValue();
            }
        }
        return null;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        MeterKey meterKey = (MeterKey) o;
        return Objects.equals(name, meterKey.name) && Arrays.equals(tags, meterKey.tags);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(name);
        result = 31 * result + Arrays.hashCode(tags);
        return result;
    }

    @Override
    public String toString() {
        return "MeterKey{" +
                "name='" + name + '\'' +
                ", tags=" + Arrays.toString(tags) +
                "}";
    }
}