package com.atlassian.diagnostics.ipd.api;

import com.atlassian.diagnostics.ipd.api.meters.config.MeterConfigBuilder;
import com.atlassian.diagnostics.ipd.api.meters.config.ProductMeterConfigBuilder;

import java.util.function.Consumer;

/**
 * Interface for managing meter configurations.
 * <p>
 * This interface allows adding configurations for specific meter key prefixes and evaluating configurations
 * for given meter keys. The configurations can be applied to meters based on their key prefixes.
 * </p>
 * See {@link #create(Consumer)} for creating a new instance of MeterConfigurations.
 * @since 5.0.0
 */
public interface MeterConfigurations {

    /**
     * Adds a configuration for a specific meter key prefix.
     * The key prefix must be made of full meter segments it wants to match. Matching prefix as `String.startsWith()` is not supported.
     *
     * <p>
     *     For example, a meter "issue.created" will apply configurations with prefix "issue" and "issue.created".
     *     However, it won't apply configuration with prefix "iss". Such configuration would apply to meters like "iss" and "iss.created".
     * </p>
     *
     * <p>
     *     The configuration is applied to all meters whose key starts with the provided prefix.
     *     The configuration is applied after the default configuration.
     * </p>
     * <p>
     *      Configuration added with prefix "" (empty string) will apply to all meters.
     * </p>
     *
     * <b>
     *     Important Note: The configuration is applied only on meter creation. Changing the configuration after the meter is created will not affect the meter.
     *     It is recommended to declare all configurations before creating any meters, otherwise meters may not have the expected configuration.
     * </b>
     *
     *
     * @param prefixMatcher the prefix of the meter key to match
     * @param meterConfig the configuration that will be applied to meters with keys matching the prefix
     * @return this MeterConfigurations instance
     */
    MeterConfigurations addMeterConfig(String prefixMatcher, Consumer<MeterConfigBuilder> meterConfig);

    /**
     * Evaluates the configuration for the given MeterKey and applies it to the provided ProductMeterConfigBuilder.
     * The default configuration is applied first, followed by any specific configurations that match the MeterKey.
     *
     * <p>
     *     This method applies configurations for each segment of the meter key, starting from the most generic to the most specific.
     *     For example, if the meter key is "issue.created" and there are configurations for "issue" and "issue.created",
     *     the configuration for "issue" will be applied first, followed by the configuration for "issue.created".
     * </p>
     *
     * @param meterKey the key of the meter to evaluate the configuration for
     * @param configBuilder the builder to apply the configuration to
     */
    void evaluateConfig(MeterKey meterKey, ProductMeterConfigBuilder configBuilder);

    /**
     * Creates a new instance of MeterConfigurations with the provided default configuration.
     *
     * @param defaultConfig the default configuration to apply to all meter configurations. The defaultConfig is always applied first, followed by any specific configurations that match the MeterKey.
     * @return a new instance of MeterConfigurations
     */
    static MeterConfigurations create(Consumer<ProductMeterConfigBuilder> defaultConfig) {
        return new MeterConfigurationsImpl(defaultConfig);
    }
}
