package com.atlassian.diagnostics;

import javax.annotation.Nonnull;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Spliterator;
import java.util.function.Consumer;

import static java.util.Optional.empty;
import static java.util.Optional.of;

/**
 * A page of items, typically returned from a service that supports pagination of results. A page of data is the result
 * of a {@link #getRequest() page request}, and provides requests that can be used to retrieve the
 * {@link #getPrevRequest() previous} and {@link #getNextRequest() next} pages (if available).
 * <p>
 * Pagination is done based on numeric offsets.
 * <p>
 * Services that return {@code Page} instances are expected to retrieve the requested results + 1 extra item. When this
 * extra value is provided to the {@link #Page(PageRequest, List) constructor}, the page can determine that
 * there is a next page of data and can provide a {@link #getNextRequest() next request}.
 * <p>
 * Any page constructed with a {@link PageRequest request} that has a {@link PageRequest#getStart() start} greater
 * than {@code 0} is assumed to have a previous page that starts with a {@link PageRequest#getStart() start offset} of
 * {@code start - limit}, or {@code 0}, whichever is greater.
 *
 * @param <T> the type the items contained in the page
 */
public class Page<T> implements Iterable<T> {

    private final PageRequest request;
    private final List<T> values;

    /**
     * @param request the request that triggered this page
     * @param values the resulting values, if a next page is available. If more values are provided tha
     */
    public Page(@Nonnull PageRequest request, @Nonnull List<T> values) {
        this.request = request;
        this.values = values;
    }

    @Nonnull
    @Override
    public Iterator<T> iterator() {
        return getValues().iterator();
    }

    @Override
    public void forEach(@Nonnull Consumer<? super T> action) {
        getValues().forEach(action);
    }

    @Nonnull
    public Optional<PageRequest> getNextRequest() {
        int limit = request.getLimit();
        if (values.size() > limit) {
            // we have more than a full page, there will be a next page
            return of(PageRequest.of(request.getStart() + limit, limit));
        }
        return empty();
    }

    @Nonnull
    public Optional<PageRequest> getPrevRequest() {
        int start = request.getStart();
        if (start > 0) {
            int limit = request.getLimit();
            return of(PageRequest.of(Math.max(0, start - limit), limit));
        }
        return empty();
    }

    @Nonnull
    public PageRequest getRequest() {
        return request;
    }

    @Nonnull
    public List<T> getValues() {
        if (values.size() <= request.getLimit()) {
            // no extra elements provided
            return values;
        }
        // extra elements(s) provided at the end of values
        return values.subList(0, request.getLimit());
    }

    public boolean isEmpty() {
        return values.isEmpty();
    }

    public int size() {
        return Math.min(request.getLimit(), values.size());
    }

    @Nonnull
    @Override
    public Spliterator<T> spliterator() {
        return getValues().spliterator();
    }
}
