package com.atlassian.diagnostics;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.diagnostics.DiagnosticsConstants.PLUGIN_NOT_DETECTED_KEY;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * Identifies the module (or more generically the 'thing in the system') and/or the plugin that was responsible
 * for the {@link Alert alert}.
 */
public class AlertTrigger {

    private final String module;
    private final String pluginKey;
    private final String pluginVersion;

    private AlertTrigger(Builder builder) {
        module = builder.module;
        pluginKey = builder.pluginKey == null ? PLUGIN_NOT_DETECTED_KEY : builder.pluginKey;
        pluginVersion = builder.pluginVersion;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AlertTrigger trigger = (AlertTrigger) o;
        return Objects.equals(module, trigger.module) &&
                Objects.equals(pluginKey, trigger.pluginKey) &&
                Objects.equals(pluginVersion, trigger.pluginVersion);
    }

    @Nonnull
    public Optional<String> getModule() {
        return ofNullable(module);
    }

    /**
     * @return the plugin key, or {@link DiagnosticsConstants#PLUGIN_NOT_DETECTED_KEY} if no plugin was detected
     */
    @Nonnull
    public String getPluginKey() {
        return pluginKey;
    }

    @Nonnull
    public Optional<String> getPluginVersion() {
        return ofNullable(pluginVersion);
    }

    @Override
    public int hashCode() {
        return Objects.hash(module, pluginKey, pluginVersion);
    }

    public static class Builder {

        private String module;
        private String pluginKey;
        private String pluginVersion;

        @Nonnull
        public AlertTrigger build() {
            return new AlertTrigger(this);
        }

        @Nonnull
        public Builder module(@Nullable String value) {
            module = trimToNull(value);
            return this;
        }

        @Nonnull
        public Builder plugin(@Nullable String key, @Nullable String version) {
            pluginKey = trimToNull(key);
            pluginVersion = trimToNull(version);
            return this;
        }
    }
}
