package com.atlassian.diagnostics;

import com.google.common.base.MoreObjects;

import javax.annotation.Nonnull;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

public class PageRequest {

    private final int start;
    private final int limit;

    private PageRequest(Builder builder) {
        start = builder.start;
        limit = builder.limit;
    }

    @Nonnull
    public static PageRequest of(int start, int limit) {
        return new Builder(limit).start(start).build();
    }

    @Nonnull
    public static PageRequest ofSize(int limit) {
        return of(0, limit);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PageRequest that = (PageRequest) o;
        return limit == that.limit && start == that.start;
    }

    public int getLimit() {
        return limit;
    }

    public int getStart() {
        return start;
    }

    @Override
    public int hashCode() {
        return Objects.hash(start, limit);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("start", start)
                .add("limit", limit)
                .toString();
    }

    public static class Builder {

        private int limit;
        private int start;

        public Builder(@Nonnull PageRequest other) {
            limit = requireNonNull(other, "other").limit;
            start = other.start;
        }

        public Builder(int limit) {
            this.limit = limit;
        }

        @Nonnull
        public PageRequest build() {
            return new PageRequest(this);
        }

        @Nonnull
        public Builder limit(int value) {
            if (limit < 0) {
                throw new IllegalArgumentException("limit must be at least 0");
            }
            limit = value;
            return this;
        }

        @Nonnull
        public Builder start(int value) {
            if (start < 0) {
                throw new IllegalArgumentException("start must be at least 0");
            }
            start = value;
            return this;
        }
    }
}
