package com.atlassian.devrel.servlet;

import com.atlassian.devrel.plugin.SoapService;
import com.atlassian.devrel.provider.SoapServiceProvider;
import com.atlassian.devrel.rest.model.RestDescriptor;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.predicate.ModuleDescriptorPredicate;
import com.atlassian.templaterenderer.TemplateRenderer;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.*;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;

public class RestBrowserServlet extends HttpServlet {
    private static final Logger log = LoggerFactory.getLogger(RestBrowserServlet.class);
    private static final String BROWSER_TEMPLATE = "/templates/browser.vm";

    private final PluginAccessor pluginAccessor;
    private final TemplateRenderer renderer;
    private final SoapServiceProvider soapServiceProvider;

    public RestBrowserServlet(PluginAccessor pluginAccessor, TemplateRenderer renderer,
                              SoapServiceProvider soapServiceProvider) {
        this.pluginAccessor = checkNotNull(pluginAccessor, "pluginAccessor");
        this.renderer = checkNotNull(renderer, "renderer");
        this.soapServiceProvider = checkNotNull(soapServiceProvider, "soapServiceProvider");
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        Map<String, Object> context = Maps.newHashMap();
        context.put("restDescriptors", getRestDescriptors());
        context.put("jsonRpcDescriptors", getJsonRpcDescriptors());

        resp.setContentType("text/html;charset=utf-8");
        renderer.render(BROWSER_TEMPLATE, context, resp.getWriter());
    }

    private List<RestDescriptor> getJsonRpcDescriptors() {
        List<RestDescriptor> jsonRpcDescriptors = Lists.newArrayList();

        Iterable<? extends SoapService> soaps = soapServiceProvider.getSoapServices();

        for (SoapService soap: soaps) {
            RestDescriptor.Builder builder = new RestDescriptor.Builder();
            Plugin plugin = soap.getPlugin();
            builder.pluginKey(plugin.getKey()).pluginCompleteKey(soap.getCompleteKey()).
                    pluginName(plugin.getName()).pluginDescription(soap.getDescription()).
                    basePath("/rpc/json-rpc").version(soap.getServicePath());
            jsonRpcDescriptors.add(builder.build());
        }

        return jsonRpcDescriptors;
    }

    private List<RestDescriptor> getRestDescriptors()
    {
        Collection<ModuleDescriptor> restServlets = pluginAccessor.getModuleDescriptors(
                new ModuleDescriptorOfClassNamePredicate("com.atlassian.plugins.rest.module.RestServletFilterModuleDescriptor"));

        List<RestDescriptor> restDescriptors = new ArrayList<RestDescriptor>(restServlets.size());

        for (ModuleDescriptor servlet : restServlets)
        {
            //com.atlassian.plugins.rest.module is exported as private and so we have to use reflection to access the concrete descriptor we want
            Class params[] = {};
            Object args[] = {};
            String basePath;
            Object apiVersion;
            String version;
            try
            {
                Method getBasePath = servlet.getClass().getMethod("getBasePath",params);
                basePath = (String)getBasePath.invoke(servlet,args);

                Method getVersion = servlet.getClass().getMethod("getVersion",params);
                apiVersion = getVersion.invoke(servlet,args);
                version = apiVersion.toString();

            } catch (Exception e)
            {
                continue;
            }


            Plugin plugin = servlet.getPlugin();

            RestDescriptor.Builder builder = new RestDescriptor.Builder().
                basePath(basePath).version(version).pluginCompleteKey(servlet.getCompleteKey()).
                pluginKey(plugin.getKey()).pluginName(plugin.getName()).pluginDescription(servlet.getDescription());
            restDescriptors.add(builder.build());
        }

        return restDescriptors;
    }

    public class ModuleDescriptorOfClassNamePredicate<T> implements ModuleDescriptorPredicate<T>
    {
        private final String moduleDescriptorClass;

        public ModuleDescriptorOfClassNamePredicate(String targetClass)
        {
            moduleDescriptorClass = targetClass;
        }

        public boolean matches(final ModuleDescriptor<? extends T> moduleDescriptor)
        {
            return (moduleDescriptor != null) && moduleDescriptor.getClass().getName().equals(moduleDescriptorClass);
        }
    }

}