package com.atlassian.db.config.password.ciphers.base64;

import com.atlassian.db.config.password.Cipher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Base64;

/**
 * Basic and default implementation of Cipher. Uses {@link Base64} to obfuscate data.
 * <p>
 * Warning!
 * Data obfuscated using this cipher are not secure. If attacker knows the obfuscated password they can decrypt it easily.
 * When using this cipher consider limiting access to the configuration file to increase security level.
 */
public class Base64Cipher implements Cipher {

    private final static Logger log = LoggerFactory.getLogger(Base64Cipher.class);

    public Base64Cipher() {
        log.debug("Initiate Base64Cipher");
    }

    /**
     * Encodes plainTextData using {@link Base64}
     *
     * @param plainTextData data in plain text which should be encrypted
     * @return data encoded using {@link Base64}
     */
    @Override
    public String encrypt(final String plainTextData) {
        log.debug("Encrypting data...");
        try {
            final String processed = encodeToString(plainTextData.getBytes());
            log.debug("Encryption done.");
            return processed;
        } catch (final Exception runtimeException) {
            log.error("Runtime Exception thrown when encrypting: " + plainTextData, runtimeException);
            throw runtimeException;
        }
    }

    /**
     * Decodes encryptedData using {@link Base64}
     *
     * @param encryptedData data which has been encrypted by using {@link #encrypt(String)}
     * @return plain text data
     */
    @Override
    public String decrypt(final String encryptedData) {
        log.debug("Decrypting data...");
        try {
            final String processed = new String(decodeFromString(encryptedData));
            log.debug("Decryption done.");
            return processed;
        } catch (final Exception runtimeException) {
            log.error("Runtime Exception thrown when decrypting: " + encryptedData, runtimeException);
            throw runtimeException;
        }
    }

    private String encodeToString(final byte[] src) {
        return Base64.getEncoder().encodeToString(src);
    }

    private byte[] decodeFromString(final String src) {
        return Base64.getDecoder().decode(src);
    }
}