package com.atlassian.db.config.password.ciphers.algorithm;

import com.atlassian.db.config.password.Cipher;
import com.atlassian.db.config.password.ciphers.algorithm.paramters.DecryptionParameters;
import com.atlassian.db.config.password.ciphers.algorithm.paramters.EncryptionParameters;
import com.atlassian.db.config.password.ciphers.algorithm.serialization.AlgorithmParametersSerializationFile;
import com.atlassian.db.config.password.ciphers.algorithm.serialization.EnvironmentVarBasedConfiguration;
import com.atlassian.db.config.password.ciphers.algorithm.serialization.SerializationFile;
import com.atlassian.db.config.password.ciphers.algorithm.serialization.SerializationFileFactory;
import com.atlassian.db.config.password.ciphers.algorithm.serialization.UniqueFilePathGenerator;
import com.google.gson.Gson;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.SerializationUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.KeyGenerator;
import javax.crypto.SealedObject;
import javax.crypto.spec.SecretKeySpec;
import java.io.Serializable;
import java.security.AlgorithmParameterGenerator;
import java.security.AlgorithmParameters;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.time.Clock;
import java.util.Base64;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.UnaryOperator;

import static com.atlassian.db.config.password.ciphers.algorithm.Algorithms.AES;
import static com.atlassian.db.config.password.ciphers.algorithm.Algorithms.DES;
import static com.atlassian.db.config.password.ciphers.algorithm.Algorithms.DESEDE;

/**
 * Advanced implementation of Cipher. Uses, specified by user, algorithm to encrypt data and stores it in a file.
 * <p>
 * It's advanced, because it operates on data stored in files, which are separate from the configuration file.
 * Therefore they can be secured.
 * <p>
 * see docs for {@link AlgorithmCipher#encrypt(String)} and {@link AlgorithmCipher#decrypt(String)}} to learn more.
 * <p>
 * Supported algorithms (in brackets key size which will be used for key generation):
 * AES/CBC/PKCS5Padding (128)
 * DES/CBC/PKCS5Padding (56)
 * DESede/CBC/PKCS5Padding (168)
 */
public class AlgorithmCipher implements Cipher {
    private static final Logger log = LoggerFactory.getLogger(AlgorithmCipher.class);

    private final Provider provider = new BouncyCastleProvider();
    private final Gson gson = new Gson();

    private final SerializationFileFactory factory;
    private Clock clock = Clock.systemUTC();
    private Function<String, String> getSystemEnv = System::getenv;

    // as this has to be visible for creating by reflection
    @SuppressWarnings("WeakerAccess")
    public AlgorithmCipher() {
        log.debug("Initiate AlgorithmCipher");
        Security.addProvider(provider);
        factory = new SerializationFileFactory();
    }

    AlgorithmCipher(final SerializationFileFactory factory, final Clock clock, final Function<String, String> getSystemEnv) {
        this.factory = factory;
        this.clock = clock;
        this.getSystemEnv = getSystemEnv;
    }

    /**
     * Creates and then saves in file {@link SealedObject} which stores encrypted data.
     * Data is encrypted using: {@link SecretKeySpec}, {@link AlgorithmParameters} and algorithm specified by user.
     * <p>
     * As parameter expects {@link EncryptionParameters} in JSON format
     * Mandatory fields in JSON:
     * <p>
     * {@link EncryptionParameters#plainTextPassword} - password in plain text
     * <p>
     * {@link EncryptionParameters#algorithm} - one of:
     * - AES/CBC/PKCS5Padding
     * - DES/CBC/PKCS5Padding
     * - DESede/CBC/PKCS5Padding
     * <p>
     * {@link EncryptionParameters#algorithmKey} - should correspond with algorithm field and be one of:
     * - AES
     * - DES
     * - DESede
     * <p>
     * Optional fields in JSON:
     * (if missing data is searched in environmental variable, if environmental variable is empty then they are generated)
     * <p>
     * {@link EncryptionParameters#algorithmParametersFilePath} - path to file which contains {@link AlgorithmParameters} stored in encoded form.
     * see {@link AlgorithmParametersSerializationFile} to check how it will be read / saved.
     * They should be generated for same algorithm as used for encryption.
     * In case parameter it's missing, path will be searched in environmental variable under key: {@link EnvironmentVarBasedConfiguration#ENV_VARIABLE_PREFIX} + java_security_AlgorithmParameters
     * In case environmental variable does not exist, they will be generated and then saved in file in encoded form, under unique name using pattern: java.security.AlgorithmParameters_[current system UTC time]
     * <p>
     * {@link EncryptionParameters#keyFilePath} - path to file which contains {@link SecretKeySpec} stored as serialized object.
     * see {@link SerializationFile} to check how it will be read /saved.
     * Should be generated using same algorithm as used for encryption.
     * In case it's missing, path will be searched in environmental variable under key: {@link EnvironmentVarBasedConfiguration#ENV_VARIABLE_PREFIX} + javax_crypto_spec_SecretKeySpec
     * In case environmental variable does not exist, key will be generated and then saved in file as serialized object, under unique name using pattern: javax.crypto.spec.SecretKeySpec_[current system UTC time]  see {@link UniqueFilePathGenerator}
     * <p>
     * {@link EncryptionParameters#outputFilesBasePath} - base path where created {@link SealedObject}, {@link SecretKeySpec} and {@link AlgorithmParameters} will be saved.
     * If not provided files will be saved in the default directory.
     * If provided, it must be ended with file separator ('/' or '\')
     *
     * @param encryptionParamsInJson {@link EncryptionParameters} in JSON format
     * @return {@link DecryptionParameters} in JSON format
     */
    @Override
    public String encrypt(final String encryptionParamsInJson) {
        log.debug("Encrypting data...");
        final String encrypted = gson.toJson(encrypt(gson.fromJson(encryptionParamsInJson, EncryptionParameters.class)));
        log.debug("Encryption done.");
        return encrypted;
    }

    /**
     * Decrypts data stored in {@link SealedObject} using {@link SecretKeySpec}.
     * <p>
     * As parameter expects {@link DecryptionParameters} in JSON format.
     * <p>
     * Optional fields in JSON:
     * (if missing data is searched in environmental variable)
     * <p>
     * {@link DecryptionParameters#serializedSealedObject} - string which contains {@link SealedObject} stored as serialized object.
     * In case it's missing, the data about {@link SealedObject} in stored by {@link DecryptionParameters#sealedObjectFilePath}.
     * <p>
     * {@link DecryptionParameters#sealedObjectFilePath} - path to file which contains {@link SealedObject} stored as serialized object.
     * see {@link SerializationFile} to check how it will be read.
     * If {@link DecryptionParameters#serializedSealedObject} is not null, this param will not be taken into account.
     * But in case when {@link DecryptionParameters#sealedObjectFilePath} and {@link DecryptionParameters#serializedSealedObject} are missing,
     * path will be searched in environmental variable under key: {@link EnvironmentVarBasedConfiguration#ENV_VARIABLE_PREFIX} + javax_crypto_SealedObject
     * <p>
     * {@link DecryptionParameters#keyFilePath} - path to file which contains {@link SecretKeySpec} stored as serialized object.
     * see {@link SerializationFile} to check how it will be read /saved.
     * In case it's missing, path will be searched in environmental variable under key: {@link EnvironmentVarBasedConfiguration#ENV_VARIABLE_PREFIX} + javax_crypto_spec_SecretKeySpec
     *
     * @param decryptionParamsInJson {@link DecryptionParameters} in JSON format
     * @return plain text password
     */
    @Override
    public String decrypt(final String decryptionParamsInJson) {
        log.debug("Decrypting data...");
        final String decrypted = decrypt(gson.fromJson(decryptionParamsInJson, DecryptionParameters.class)).getPlainTextPassword();
        log.debug("Decryption done.");
        return decrypted;
    }

    private EncryptionParameters decrypt(final DecryptionParameters dataToDecrypt) {
        try {
            final String plainTextPassword = (String) getEncryptedPassword(dataToDecrypt)
                    .getObject(tryFromParamsThenEnvThenThrow(dataToDecrypt.getKeyFilePath(), SecretKeySpec.class));

            return new EncryptionParameters.Builder()
                    .setPlainTextPassword(plainTextPassword)
                    .build();

        } catch (final Exception e) {
            log.error("Runtime Exception thrown when decrypting: " + dataToDecrypt, e);
            throw new RuntimeException(e);
        }
    }

    protected SealedObject getEncryptedPassword(final DecryptionParameters dataToDecrypt) {
        return Optional.ofNullable(dataToDecrypt.getSerializedSealedObject())
                .map(this::base64ToObject)
                .orElseGet(() -> tryFromParamsThenEnvThenThrow(dataToDecrypt.getSealedObjectFilePath(), SealedObject.class));
    }

    protected DecryptionParameters encrypt(final EncryptionParameters parameters) {
        try {
            final UnaryOperator<String> buildPath = (fileName) -> buildFilePath(parameters.getOutputFilesBasePath(), fileName);

            final AlgorithmParameters algorithmParameters = getAlgorithmParameters(parameters, buildPath);

            final KeyWithPath keyWithPath = getKeyOrGenerateNewAndGet(parameters, buildPath);
            final SecretKeySpec key = keyWithPath.getSecretKeySpec();

            // encrypt password
            final javax.crypto.Cipher cipher = javax.crypto.Cipher.getInstance(parameters.getAlgorithm(), provider);
            cipher.init(javax.crypto.Cipher.ENCRYPT_MODE, key, algorithmParameters);
            final SealedObject encryptedPass = new SealedObject(parameters.getPlainTextPassword(), cipher);

            final String pathToSealedObject;
            final String base64encoded;
            if (parameters.isSaveSealedObjectToSeparateFile()) {
                //generate path for encrypted pass and save
                pathToSealedObject = buildPath.apply(generateFileName(encryptedPass.getClass().getName()));
                factory.getSerializationFile(pathToSealedObject).createFileAndSave(encryptedPass);
                base64encoded = null;
            } else {
                base64encoded = objectToBase64(encryptedPass);
                pathToSealedObject = null;
            }

            return new DecryptionParameters.Builder()
                    .setSealedObjectFilePath(pathToSealedObject)
                    .serializedSealedObject(base64encoded)
                    .setKeyFilePath(keyWithPath.getPath())
                    .build();

        } catch (final Exception e) {
            log.error("Exception thrown when encrypting: " + parameters, e);
            throw new RuntimeException(e);
        }
    }

    private AlgorithmParameters getAlgorithmParameters(final EncryptionParameters parameters, final UnaryOperator<String> buildPath) {
        final AlgorithmParameters algorithmParameters;
        // get path from params, then try from env
        String algParamsPath = ObjectUtils.firstNonNull(
                parameters.getAlgorithmParametersFilePath(),
                getFromEnv(AlgorithmParameters.class.getName()));

        if (algParamsPath == null) {
            // if path is not present, generate alg params, file name(/path) for it and then save
            algorithmParameters = generateAlgorithmParameters(parameters.getAlgorithmKey());
            if (parameters.isSaveAlgorithmParametersToSeparateFile()) {
                algParamsPath = buildPath.apply(generateFileName(AlgorithmParameters.class.getName()));
                factory.getAlgorithmParametersSerializationFile(algParamsPath).createFileAndSave(algorithmParameters);
                log.debug("Name of generated file with algorithm params used for encryption: {}", algParamsPath);
            } else {
                log.debug("Generation of file for algorithm params has been skipped");
            }
        } else {
            // read alg params if file path is present
            algorithmParameters = factory.getAlgorithmParametersSerializationFile(algParamsPath).read(parameters.getAlgorithmKey());
        }
        return algorithmParameters;
    }

    private KeyWithPath getKeyOrGenerateNewAndGet(EncryptionParameters parameters, UnaryOperator<String> buildPath) {
        final SecretKeySpec key;
        // get path from params, then try from env
        String keyPath = ObjectUtils.firstNonNull(
                parameters.getKeyFilePath(),
                getFromEnv(SecretKeySpec.class.getName()));

        if (keyPath == null) {
            // if path is not present, generate key, file name(/path) for it and then save
            key = generateSecretKey(parameters.getAlgorithmKey());
            keyPath = buildPath.apply(generateFileName(SecretKeySpec.class.getName()));
            factory.getSerializationFile(keyPath).createFileAndSave(key);
        } else {
            // read key if file path is present
            key = factory.getSerializationFile(keyPath).read(SecretKeySpec.class);
        }

        return new KeyWithPath(key, keyPath);
    }

    private SecretKeySpec generateSecretKey(final String algorithmKey) {
        try {
            final KeyGenerator keyGen = KeyGenerator.getInstance(algorithmKey, provider);
            keyGen.init(getKeySize(algorithmKey));
            return (SecretKeySpec) keyGen.generateKey();
        } catch (final Exception e) {
            log.error("Exception thrown when generating key for algorithm: " + algorithmKey, e);
            throw new RuntimeException(e);
        }
    }

    private AlgorithmParameters generateAlgorithmParameters(final String algorithmKey) {
        try {
            final AlgorithmParameterGenerator algorithmParameterGenerator =
                    AlgorithmParameterGenerator.getInstance(algorithmKey, provider);
            algorithmParameterGenerator.init(new SecureRandom().nextInt(), new SecureRandom());
            return algorithmParameterGenerator.generateParameters();
        } catch (final Exception e) {
            log.error("Exception thrown when generating algorithm parameters for algorithm: " + algorithmKey, e);
            throw new RuntimeException(e);
        }
    }

    private int getKeySize(final String algorithmKey) {
        switch (algorithmKey) {
            case AES:
                return 128;
            case DES:
                return 56;
            case DESEDE:
                return 168;
            default:
                return 24;
        }
    }

    private String getFromEnv(final String objectClassName) {
        final EnvironmentVarBasedConfiguration environmentVarBasedConfiguration = new EnvironmentVarBasedConfiguration(objectClassName, getSystemEnv);
        return environmentVarBasedConfiguration.getFromEnv();
    }

    private String generateFileName(final String objectClassName) {
        return new UniqueFilePathGenerator(objectClassName, clock).generateName();
    }

    private String buildFilePath(final String basePath, final String relativePath) {
        return Optional.of(relativePath)
                .map(p -> Optional.ofNullable(basePath).orElse("") + p)
                .orElse(null);
    }

    private SealedObject base64ToObject(String base64) {
        final byte[] decoded = Base64.getDecoder().decode(base64);
        return SerializationUtils.deserialize(decoded);
    }

    private String objectToBase64(Serializable obj) {
        byte[] serializedBytes = SerializationUtils.serialize(obj);
        return Base64.getEncoder().encodeToString(serializedBytes);
    }

    private <T> T tryFromParamsThenEnvThenThrow(final String filePathFromParams, final Class<T> clz) {
        final String keyFilePath = Optional.ofNullable(filePathFromParams)
                .orElseGet(() -> Optional.ofNullable(getFromEnv(clz.getName()))
                        .orElseThrow(() -> new IllegalArgumentException("Missing file path for: " + clz.getName())));
        return factory.getSerializationFile(keyFilePath).read(clz);
    }
}
