package com.atlassian.data

import com.atlassian.data.utils.RandomUtils
import java.util.function.BiConsumer

class ValuesWithFrequencies<T> : BiConsumer<T, Long> {
    private val valuesAndFreqs: ArrayList<ValueAndFrequency<T>> = ArrayList()

    private var totalSamples: Long = 0

    override fun accept(value: T, frequency: Long) {
        totalSamples += frequency
        valuesAndFreqs.add(ValueAndFrequency(value, totalSamples))
    }

    fun pickRandom(): T {
        val index = randomIndex()
        return valuesAndFreqs[index].value
    }

    fun pickRandomTryToAvoid(excluded: T): T {
        val index = randomIndex()
        val value = valuesAndFreqs[index].value
        if (value != excluded) {
            return value
        }
        return when {
            index>0 -> {
                valuesAndFreqs[index-1].value
            }
            index < (valuesAndFreqs.size-1) -> {
                valuesAndFreqs[index+1].value
            }
            else -> {
                value
            }
        }
    }

    private fun randomIndex(): Int {
        val frequency = RandomUtils.nextLong(totalSamples)
        return frequencyToIndex(frequency)
    }

    fun frequencyToIndex(value: Long): Int {
        if (value>totalSamples) {
            throw IllegalArgumentException("$value is out of range, maximum value is $totalSamples")
        }
        val binarySearch = valuesAndFreqs.binarySearch { it.cumulativeFrequency.compareTo(value) }
        if (binarySearch<0) {
            return -binarySearch-1
        }
        return binarySearch
    }

    fun hasValue(value: T): Boolean {
        return valuesAndFreqs.any { it.value == value }
    }
}

private class ValueAndFrequency<T>(val value : T, val cumulativeFrequency: Long)
