package com.atlassian.crowd.embedded.spi;

import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.util.BatchResult;
import com.atlassian.crowd.util.BoundedCount;
import com.google.common.collect.ListMultimap;

import java.util.Collection;
import java.util.List;
import java.util.Set;

public interface MembershipDao {
    /**
     * Determines whether the user is a direct member of the group.
     *
     * @param directoryId the directory to perform the operation
     * @param userName    user
     * @param groupName   group
     * @return true if the user is a direct member of the group
     */
    boolean isUserDirectMember(long directoryId, String userName, String groupName);

    /**
     * Determines whether the group is a direct member of the (supposedly) parent group.
     *
     * @param directoryId the directory to perform the operation
     * @param childGroup  child group
     * @param parentGroup parent group
     * @return true if the group is a direct member of the (supposedly) parent group
     */
    boolean isGroupDirectMember(long directoryId, String childGroup, String parentGroup);

    /**
     * Adds user as a member of group.
     *
     * @param directoryId the directory to perform the operation
     * @param userName    user
     * @param groupName   group
     * @throws UserNotFoundException            if the user does not exist
     * @throws GroupNotFoundException           if the group does not exist
     * @throws MembershipAlreadyExistsException if the user is already a direct member of the group
     */
    void addUserToGroup(long directoryId, String userName, String groupName)
            throws UserNotFoundException, GroupNotFoundException, MembershipAlreadyExistsException;

    /**
     * Bulk adds the given user to all the given groups. Groups that user is already a direct member of are reported as
     * failures. Group names that cannot be found in the database are also reported as failures.
     *
     * Implementations must make sure that changes in bulk methods such as this are immediately visible to other bulk
     * methods. For example, if this is run in a transaction, either that transaction must be committed when this method
     * returns, or all other bulk method implementations must guarantee to reuse the same transaction.
     *
     * @param directoryId the id of the directory to which user and groups belong
     * @param username    username of the user to whom we add groups
     * @param groupNames  names of the group to add to a user. The groups need to exist in the directory
     * @return result of the bulk operation containing successful and failed entities
     * @throws UserNotFoundException when user with a given {@code username} does not exist in the directory with
     *                               {@code directoryId} id
     */
    BatchResult<String> addUserToGroups(long directoryId, String username, Set<String> groupNames) throws UserNotFoundException;

    /**
     * Bulk adds all the given users into the given group. Users that are already direct members of the group are
     * reported as failures. Usernames that cannot be found in the database are also reported as failures.
     *
     * Implementations must make sure that changes in bulk methods such as this are immediately visible to other bulk
     * methods. For example, if this is run in a transaction, either that transaction must be committed when this method
     * returns, or all other bulk method implementations must guarantee to reuse the same transaction.
     *
     * @param directoryId the directory to perform the operation
     * @param userNames   the collection of users
     * @param groupName   name of the group
     * @return result containing both successful (membership successfully added) and failed users (unknown username,
     * or already existing membership)
     * @throws GroupNotFoundException if the group does not exist
     */
    BatchResult<String> addAllUsersToGroup(long directoryId, Collection<String> userNames, String groupName) throws GroupNotFoundException;

    /**
     * Adds group as a child of the (supposedly) parent group.
     *
     * @param directoryId the directory to perform the operation
     * @param childGroup  the (supposedly) child group
     * @param parentGroup parent group
     * @throws GroupNotFoundException           if either child or parent group is not found
     * @throws MembershipAlreadyExistsException if the child group is already a child of the parent group
     */
    void addGroupToGroup(long directoryId, String childGroup, String parentGroup)
            throws GroupNotFoundException, MembershipAlreadyExistsException;

    /**
     * Removes user as a member of the given group.
     *
     * @param directoryId the directory to perform the operation
     * @param userName    user
     * @param groupName   group
     * @throws UserNotFoundException       if the user does not exist
     * @throws GroupNotFoundException      if the group does not exist
     * @throws MembershipNotFoundException if the user is not a member of the said group
     */
    void removeUserFromGroup(long directoryId, String userName, String groupName) throws UserNotFoundException, GroupNotFoundException, MembershipNotFoundException;

    /**
     * Removes group from the parent group.
     *
     * @param directoryId the directory to perform the operation
     * @param childGroup  child group
     * @param parentGroup parent group
     * @throws GroupNotFoundException      if either child or parent group does not exist
     * @throws MembershipNotFoundException if the membership relationship between the child and parent group does not exist
     */
    void removeGroupFromGroup(long directoryId, String childGroup, String parentGroup) throws GroupNotFoundException, MembershipNotFoundException;

    /**
     * Search for memberships by the given criteria.
     *
     * @param directoryId the directory to perform the operation
     * @param query       criteria
     * @return List (can be empty but never null) of memberships which match the given criteria.
     * Results are ordered by entity name, case-insensitive.
     */
    <T> List<T> search(long directoryId, MembershipQuery<T> query);

    /**
     * Search for memberships by the given criteria. Groups results by {@link MembershipQuery#getEntityNamesToMatch()}.
     *
     * @param directoryId the directory to perform the operation
     * @param query       criteria
     * @return List (can be empty but never null) of memberships which match the given criteria.
     * Results are ordered by entity name, case-insensitive.
     */
    <T> ListMultimap<String, T> searchGroupedByName(long directoryId, MembershipQuery<T> query);

    /**
     * <p>Count the direct members of a group with a recommended upper bound on the count.</p>
     *
     * <p>The upper bound is there so that, in directories that do not support efficient counting of memberships, less work
     * may be performed.</p>
     *
     * <p><strong>Important:</strong> Please note that any directory wishing to implement this method should make it as efficient
     * as possible. Such that an administrator could land on a page that showed ~20 groups, run this method once per group,
     * and it would still return results as efficiently as possible.</p>
     *
     * @param directoryId       the directory to perform the operation
     * @param groupName         the name of the group to search for
     * @param potentialMaxCount a suggested minimum number of elements for the implementation to count. The implementation
     *                          will only use this number if it can get efficiency benefits; otherwise this number may be ignored.
     * @return A bounded count of the number of memberships in the given group for the provided directory. If the group does not exist
     * then {@link com.atlassian.crowd.util.BoundedCount#exactly(long)} 0 will be returned.
     */
    BoundedCount countDirectMembersOfGroup(long directoryId, String groupName, int potentialMaxCount);
}
