package com.atlassian.crowd.embedded.spi;

import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.InternalDirectoryGroup;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.util.BatchResult;

import java.util.List;
import java.util.Map;
import java.util.Set;

public interface GroupDao {
    /**
     * Finds group by name.
     *
     * @param directoryId the ID of the directory to look for group
     * @param name        group name
     * @return group
     * @throws GroupNotFoundException if the group does not exist
     */
    InternalDirectoryGroup findByName(long directoryId, String name) throws GroupNotFoundException;

    /**
     * Finds group by name. This is different from {@link #findByName(long, String)} in that it also returns
     * the group attributes associated with the retrieved group.
     *
     * @param directoryId the ID of the directory to look for group
     * @param name        group name
     * @return group with attributes
     * @throws GroupNotFoundException if the group does not exist
     */
    GroupWithAttributes findByNameWithAttributes(long directoryId, String name) throws GroupNotFoundException;

    /**
     * Adds a new group.
     *
     * @param group group
     * @return the added group
     * @throws DirectoryNotFoundException if the directory specified in group object does not exist
     * @throws InvalidGroupException      if a group with the same directory and name (case-insensitive) already exists
     */
    Group add(Group group) throws DirectoryNotFoundException, InvalidGroupException;

    /**
     * Add a new local group. A local group is a group that does not exist in the remote directory.
     * The implementation must take into account and also persist the fact that the group is only local.
     *
     * @param group group
     * @return the added group
     * @throws DirectoryNotFoundException if the directory specified in group object does not exist
     * @throws InvalidGroupException      if a group with the same directory and name (case-insensitive) already exists
     */
    Group addLocal(Group group) throws DirectoryNotFoundException, InvalidGroupException;

    /**
     * Updates group.
     *
     * @param group group
     * @return the updated group
     * @throws GroupNotFoundException if the group does not exist
     */
    Group update(Group group) throws GroupNotFoundException;

    /**
     * Renames group.
     *
     * @param group   group
     * @param newName the new name
     * @return group with new name
     * @throws GroupNotFoundException if the group does not exist
     * @throws InvalidGroupException  if a group with the same directory and name (case-insensitive) already exists
     */
    Group rename(Group group, String newName) throws GroupNotFoundException, InvalidGroupException;

    /**
     * Stores attributes into group.
     * Any existing attributes matching the supplied attribute keys will be replaced.
     *
     * @param group      group
     * @param attributes attributes
     * @throws GroupNotFoundException if the group does not exist
     */
    void storeAttributes(Group group, Map<String, Set<String>> attributes) throws GroupNotFoundException;

    /**
     * Remove the attribute from the group. Does nothing if the attribute doesn't exist.
     *
     * @param group         group
     * @param attributeName attribute to be removed
     * @throws GroupNotFoundException if the group does not exist
     */
    void removeAttribute(Group group, String attributeName) throws GroupNotFoundException;

    /**
     * Removes group.
     *
     * @param group group
     * @throws GroupNotFoundException if the group does not exist
     */
    void remove(Group group) throws GroupNotFoundException;

    /**
     * Searches for group based on the given criteria.
     *
     * @param directoryId directory to perform the search operation on
     * @param query       criteria
     * @return list (could be empty) of groups which match the criteria
     */
    <T> List<T> search(long directoryId, EntityQuery<T> query);

    /**
     * Bulk add of groups. Will only add remote groups (ie. isLocal=false)
     *
     * Implementations must make sure that changes in bulk methods such as this are immediately visible to other bulk
     * methods. For example, if this is run in a transaction, either that transaction must be committed when this method
     * returns, or all other bulk method implementations must guarantee to reuse the same transaction.
     *
     * @param groups to be added
     * @return a list of Groups that <b>failed</b> to be added
     * @throws com.atlassian.crowd.exception.DirectoryNotFoundException if the directory cannot be found
     */
    BatchResult<Group> addAll(Set<? extends Group> groups) throws DirectoryNotFoundException;

    /**
     * Bulk remove all the given groups.
     *
     * Implementations must make sure that changes in bulk methods such as this are immediately visible to other bulk
     * methods. For example, if this is run in a transaction, either that transaction must be committed when this method
     * returns, or all other bulk method implementations must guarantee to reuse the same transaction.
     *
     * @param directoryId directory to perform the operation
     * @param groupNames  groups to be removed
     * @return batch result containing successes (removed groups) and failures (groups which were not removed)
     */
    BatchResult<String> removeAllGroups(long directoryId, Set<String> groupNames);

    /**
     * Return all group externalId in the given directory.
     * If a group's externalId is {@code null} or an empty String it won't be included.
     *
     * @param directoryId the ID of the directory
     * @return set containing all externalIds with nulls filtered out
     * @throws DirectoryNotFoundException when directory with given id does not exist
     */
    Set<String> getAllExternalIds(long directoryId) throws DirectoryNotFoundException;

    /**
     * Return number of groups in given directory.
     *
     * @param directoryId the ID of the directory
     * @return group count
     * @throws DirectoryNotFoundException when directory with given id does not exist
     */
    long getGroupCount(long directoryId) throws DirectoryNotFoundException;

    /**
     * Return names of all local groups in given directory.
     *
     * @param directoryId the ID of the directory
     * @return names of all local groups in given directory
     * @throws DirectoryNotFoundException when directory with given id does not exist
     */
    Set<String> getLocalGroupNames(long directoryId) throws DirectoryNotFoundException;

    /**
     * Searches the specified directory for group names of groups with the specified external ids
     *
     * @param directoryId the id of the directory to search
     * @param externalIds external ids of the groups to find
     * @return a map from external ids to group names
     */
    Map<String, String> findByExternalIds(long directoryId, Set<String> externalIds);

    /**
     * Searches the specified directory for externalIds of groups with the specified names
     *
     * @param directoryId the id of the directory to search
     * @param groupNames  names of the groups to find
     * @return a map from group names to external ids
     */
    Map<String, String> findExternalIdsByNames(long directoryId, Set<String> groupNames);

    /**
     * Gets the count of external groups for a directory
     *
     * @param directoryId the directory to count groups for
     * @return count of external groups
     */
    long getExternalGroupCount(long directoryId) throws DirectoryNotFoundException;
}