package com.atlassian.crowd.embedded.spi;

import com.atlassian.annotations.ExperimentalSpi;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.model.directory.DirectorySynchronisationStatus;

import java.util.Collection;
import java.util.Optional;
import java.util.Set;

/**
 * A DAO to store synchronisation progress and results
 */
@ExperimentalSpi
public interface DirectorySynchronisationStatusDao {
    /**
     * @return the status of the latest active synchronization for the given directory if present
     */
    Optional<DirectorySynchronisationStatus> findActiveForDirectory(long directoryId);

    /**
     * @return the status of the latest completed synchronization for the given directory if present
     */
    Optional<DirectorySynchronisationStatus> findLastForDirectory(long directoryId);

    /**
     * Saves a new synchronisation status
     * @return the newly saved status
     */
    DirectorySynchronisationStatus add(DirectorySynchronisationStatus status);

    /**
     * Updates an existing directory status
     * @return the updated status. Must have an id set
     * @throws ObjectNotFoundException if a status with the given id doesn't exist
     */
    DirectorySynchronisationStatus update(DirectorySynchronisationStatus status) throws ObjectNotFoundException;

    /**
     * Removes all stored status information for the directory with the specified id
     * @param directoryId the directory id to remove for
     * @return the number of entries removed
     */
    long removeStatusesForDirectory(Long directoryId);

    /**
     * Removes all stored status information
     * @return the number of entries removed
     */
    long removeAll();

    /**
     * Removes all stored status information for the directory except the indicated one
     * @param directoryId the directory id
     * @param statusId the id of the status to keep
     * @return the number of entries removed
     */
    long removeAllExcept(long directoryId, int statusId);

    /**
     * @param nodesIds the set of node ids to filter on
     * @return entries for synchronisations that are indicated as active but not owned by any of the nodes ids
     */
    Collection<DirectorySynchronisationStatus> findActiveSyncsWhereNodeIdNotIn(Set<String> nodesIds);
}