package com.atlassian.crowd.embedded.impl;

import com.google.common.base.Function;
import com.google.common.collect.Multimap;
import org.apache.commons.lang3.StringUtils;

import java.util.Collection;
import java.util.HashSet;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Predicate;

public final class IdentifierUtils {
    /**
     * Not for instantiation.
     */
    private IdentifierUtils() {
    }

    private static Locale IDENTIFIER_COMPARE_LOCALE;

    static {
        prepareIdentifierCompareLocale();
    }

    public static void prepareIdentifierCompareLocale() {
        // This system property defines the language rule
        // by which all the identifiers in crowd are normalized for comparison.
        final String preferredLang = System.getProperty("crowd.identifier.language");
        IDENTIFIER_COMPARE_LOCALE = StringUtils.isNotBlank(preferredLang) ? new Locale(preferredLang) : Locale.ENGLISH;
    }

    /**
     * Converts the given identifier string to lowercase.
     * The rule of conversion is subject to the language defined in crowd.identifier.language system property.
     *
     * @param identifier the identifier string, null allowed.
     * @return lowercase identifier.
     */
    public static String toLowerCase(String identifier) {
        return identifier == null ? null : identifier.toLowerCase(IDENTIFIER_COMPARE_LOCALE);
    }

    /**
     * Converts the given set identifier string to lowercase.
     * The rule of conversion is subject to the language defined in crowd.identifier.language system property.
     *
     * @param identifiers the identifier strings, nulls allowed.
     * @return set of lowercase identifier.
     */
    public static Set<String> toLowerCase(Collection<? extends String> identifiers) {
        Set<String> lowerCased = new HashSet<>(identifiers.size());
        for (String identifier : identifiers) {
            lowerCased.add(toLowerCase(identifier));
        }
        return lowerCased;
    }

    /**
     * @param identifiers the identifier strings, nulls allowed.
     * @return {@link Predicate} returning whether given identifier is in {@code identifiers}, ignoring case
     */
    public static Predicate<String> containsIdentifierPredicate(Collection<String> identifiers) {
        if (identifiers.isEmpty()) {
            return s -> false;
        }
        final Set<String> lowerCaseIds = toLowerCase(identifiers);
        return (name) -> lowerCaseIds.contains(toLowerCase(name));
    }

    /**
     * Converts the two given identifier strings to lowercase and compare them.
     * The rule of conversion is subject to the language defined in crowd.identifier.language system property.
     *
     * @param identifier1 identifier. Must not be null
     * @param identifier2 identifier. Must not be null
     * @return comparison result similar to as {@link java.util.Comparator#compare(Object, Object)}}
     */
    public static int compareToInLowerCase(String identifier1, String identifier2) {
        return toLowerCase(identifier1).compareTo(toLowerCase(identifier2));
    }

    /**
     * Converts the two given identifier strings to lowercase and check for equality.
     * The rule of conversion is subject to the language defined in crowd.identifier.language system property.
     *
     * @param identifier1 identifier. Can be null
     * @param identifier2 identifier. Can be null
     * @return true if equal, otherwise false.
     */
    public static boolean equalsInLowerCase(String identifier1, String identifier2) {
        if (identifier1 == null) {
            return identifier2 == null;
        } else {
            return (identifier2 != null) && (compareToInLowerCase(identifier1, identifier2) == 0);
        }
    }

    /**
     * Function of {@link #toLowerCase(String)} method.
     */
    public static final Function<String, String> TO_LOWER_CASE = IdentifierUtils::toLowerCase;

    /**
     * @param s a non-<code>null</code> identifier
     * @return <code>true</code> if this identifier starts or ends with white space
     */
    public static boolean hasLeadingOrTrailingWhitespace(String s) {
        return !s.equals(s.trim());
    }

    /**
     * Returns function that checks presence of an entry in the original {@code multimap}, ignoring string casing.
     * Works like case insensitive version of {@link Multimap#containsEntry(Object, Object)}.
     * @param multimap original case sensitive map
     * @return function that checks presence of an entry in the original {@code multimap}, ignoring string casing
     * @see Multimap#containsEntry(Object, Object)
     */
    public static BiPredicate<String, String> containsIdentifierBiPredicate(final Multimap<String, String> multimap) {
        final Map<String, Predicate<String>> predicateMap = new HashMap<>();
        multimap.asMap().forEach((k, v) -> predicateMap.put(toLowerCase(k), containsIdentifierPredicate(v)));
        return (key, value) -> predicateMap.getOrDefault(toLowerCase(key), name -> false).test(value);
    }
}
